"use server";

import dns from "node:dns/promises";

export interface DnsResult {
    provider: string;
    flag: string;
    location: string;
    status: "ok" | "error";
    value: string;
    ttl: string;
    time: string;
}

const formatResult = (provider: string, flag: string, location: string, data: any, startTime: number): DnsResult => {
    const time = Math.floor(performance.now() - startTime) + "ms";

    if (!data) {
        return { provider, flag, location, status: "error", value: "Resolution Failed", ttl: "-", time };
    }

    if (data.Status !== undefined && data.Status !== 0) {
        let errorMsg = `Error (Code: ${data.Status})`;
        if (data.Status === 3) errorMsg = "NXDOMAIN (Not Found)";
        if (data.Status === 2) errorMsg = "SERVFAIL";
        if (data.Status === 5) errorMsg = "REFUSED";
        return { provider, flag, location, status: "error", value: errorMsg, ttl: "-", time };
    }

    if (data.Answer && data.Answer.length > 0) {
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        const answers = data.Answer.map((a: any) => a.data).join(", ");
        const ttl = data.Answer[0].TTL?.toString() || "-";
        return { provider, flag, location, status: "ok", value: answers, ttl, time };
    }

    return { provider, flag, location, status: "error", value: "No Records Found", ttl: "-", time };
};

const fetchWithTimeout = async (url: string, options: RequestInit = {}, timeout = 4000) => {
    const controller = new AbortController();
    const id = setTimeout(() => controller.abort(), timeout);
    try {
        const response = await fetch(url, {
            ...options,
            signal: controller.signal
        });
        clearTimeout(id);
        return response;
    } catch (error) {
        clearTimeout(id);
        throw error;
    }
};

export async function checkDnsPropagation(domain: string, type: string): Promise<DnsResult[]> {
    // 1. Google
    const checkGoogle = async (): Promise<DnsResult> => {
        const start = performance.now();
        try {
            const res = await fetchWithTimeout(`https://dns.google/resolve?name=${domain}&type=${type}`, { cache: 'no-store' });
            if (!res.ok) throw new Error(res.statusText);
            const data = await res.json();
            return formatResult("Google DNS", "🇺🇸", "United States", data, start);
        } catch (e) {
            console.error("Google DNS Error:", e);
            return formatResult("Google DNS", "🇺🇸", "United States", null, start);
        }
    };

    // 2. Cloudflare
    const checkCloudflare = async (): Promise<DnsResult> => {
        const start = performance.now();
        try {
            const res = await fetchWithTimeout(`https://cloudflare-dns.com/dns-query?name=${domain}&type=${type}`, {
                headers: { 'Accept': 'application/dns-json' },
                cache: 'no-store'
            });
            if (!res.ok) throw new Error(res.statusText);
            const data = await res.json();
            return formatResult("Cloudflare", "🌐", "Global / Anycast", data, start);
        } catch (e) {
            console.error("Cloudflare DNS Error:", e);
            return formatResult("Cloudflare", "🌐", "Global / Anycast", null, start);
        }
    };

    // 3. AliDNS
    const checkAli = async (): Promise<DnsResult> => {
        const start = performance.now();
        try {
            const res = await fetchWithTimeout(`https://dns.alidns.com/resolve?name=${domain}&type=${type}`, { cache: 'no-store' });
            if (!res.ok) throw new Error(res.statusText);
            const data = await res.json();
            return formatResult("AliDNS", "🇨🇳", "China", data, start);
        } catch (e) {
            console.error("AliDNS Error:", e);
            return formatResult("AliDNS", "🇨🇳", "China", null, start);
        }
    };

    // 4. Quad9
    const checkQuad9 = async (): Promise<DnsResult> => {
        const start = performance.now();
        try {
            const res = await fetchWithTimeout(`https://dns.quad9.net:5053/dns-query?name=${domain}&type=${type}`, {
                method: 'GET',
                headers: { 'Accept': 'application/dns-json' },
                cache: 'no-store'
            });
            if (!res.ok) throw new Error(res.statusText + " " + res.status);
            const data = await res.json();
            return formatResult("Quad9", "🇨🇭", "Europe/Global", data, start);
        } catch (e) {
            // Quad9 often timeouts on non-standard ports in some environments
            return formatResult("Quad9", "🇨🇭", "Europe/Global", null, start);
        }
    };

    // 5. AdGuard
    const checkAdGuard = async (): Promise<DnsResult> => {
        const start = performance.now();
        try {
            const res = await fetchWithTimeout(`https://dns.adguard.com/resolve?name=${domain}&type=${type}`, { cache: 'no-store' });
            if (!res.ok) throw new Error(res.statusText + " " + res.status);
            const data = await res.json();
            return formatResult("AdGuard", "🇨🇾", "Cyprus", data, start);
        } catch {
            return formatResult("AdGuard", "🇨🇾", "Cyprus", null, start);
        }
    };

    // 6. Local System (with timeout race)
    const checkLocal = async (): Promise<DnsResult> => {
        const start = performance.now();
        try {
            // Create a timeout promise
            const timeoutPromise = new Promise((_, reject) =>
                setTimeout(() => reject(new Error("Timeout")), 4000)
            );

            // Race between resolve and timeout
            const addresses = await Promise.race([
                dns.resolve(domain, type),
                timeoutPromise
            ]) as string[] | object[];

            const rows = Array.isArray(addresses) ? addresses.map(a => typeof a === 'object' ? JSON.stringify(a) : a) : [addresses];
            const value = rows.join(", ");

            return {
                provider: "Local System",
                flag: "🖥️",
                location: "Your Host",
                status: "ok",
                value: value as string,
                ttl: "-",
                time: Math.floor(performance.now() - start) + "ms"
            };
        } catch (error) {
            // Fallback to lookup if resolve fails or times out (though lookup also takes time, we give it a quick try)
            if (type === 'A' || type === 'AAAA') {
                try {
                    // Very short timeout for fallback
                    const lookupPromise = dns.lookup(domain, { family: type === 'A' ? 4 : 6, all: true });
                    const lookupRes = await Promise.race([
                        lookupPromise,
                        new Promise((_, r) => setTimeout(() => r(new Error("Timeout")), 1000))
                    ]) as { address: string }[];

                    return {
                        provider: "Local System",
                        flag: "🖥️",
                        location: "Your Host (OS)",
                        status: "ok",
                        value: lookupRes.map(r => r.address).join(", "),
                        ttl: "-",
                        time: Math.floor(performance.now() - start) + "ms"
                    };
                } catch { }
            }

            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            const err = error as any;
            let errorMsg = err.code || err.message || "Resolution Failed";
            if (err.code === "ECONNREFUSED") errorMsg = "Unreachable";
            if (err.message === "Timeout") errorMsg = "Timed Out";

            return {
                provider: "Local System",
                flag: "🖥️",
                location: "Your Host",
                status: "error",
                value: errorMsg,
                ttl: "-",
                time: Math.floor(performance.now() - start) + "ms"
            };
        }
    };

    const results = await Promise.all([
        checkGoogle(),
        checkCloudflare(),
        checkAli(),
        checkQuad9(),
        checkAdGuard(),
        checkLocal()
    ]);

    return results;
}
