import { NextResponse } from 'next/server';

// List of ~50 Common RBLs/DNSBLs
const RBL_DATABASE = [
    { host: 'zen.spamhaus.org', name: 'Spamhaus ZEN', category: 'Critical' },
    { host: 'b.barracudacentral.org', name: 'Barracuda', category: 'Critical' },
    { host: 'bl.spamcop.net', name: 'SpamCop', category: 'Critical' },
    { host: 'dnsbl.sorbs.net', name: 'SORBS Aggregate', category: 'Reputation' },
    { host: 'spam.dnsbl.sorbs.net', name: 'SORBS Spam', category: 'Reputation' },
    { host: 'http.dnsbl.sorbs.net', name: 'SORBS HTTP', category: 'Reputation' },
    { host: 'socks.dnsbl.sorbs.net', name: 'SORBS SOCKS', category: 'Reputation' },
    { host: 'misc.dnsbl.sorbs.net', name: 'SORBS Misc', category: 'Reputation' },
    { host: 'smtp.dnsbl.sorbs.net', name: 'SORBS SMTP', category: 'Reputation' },
    { host: 'web.dnsbl.sorbs.net', name: 'SORBS Web', category: 'Reputation' },
    { host: 'cbl.abuseat.org', name: 'Abuseat CBL', category: 'Malware' },
    { host: 'bl.mailspike.net', name: 'Mailspike BL', category: 'Reputation' },
    { host: 'psbl.surriel.com', name: 'PSBL', category: 'Reputation' },
    { host: 'ubl.unsubscore.com', name: 'UBL', category: 'Reputation' },
    { host: 'dnsbl-1.uceprotect.net', name: 'UCEPROTECT Level 1', category: 'Reputation' },
    { host: 'dnsbl-2.uceprotect.net', name: 'UCEPROTECT Level 2', category: 'Reputation' },
    { host: 'dnsbl-3.uceprotect.net', name: 'UCEPROTECT Level 3', category: 'Reputation' },
    { host: 'db.wpbl.info', name: 'WPBL', category: 'Reputation' },
    { host: 'access.redhawk.org', name: 'Redhawk', category: 'Reputation' },
    { host: 'all.s5h.net', name: 'S5H', category: 'Reputation' },
    { host: 'blacklist.woody.ch', name: 'Woody', category: 'Reputation' },
    { host: 'bogons.cymru.com', name: 'Cymru Bogons', category: 'General' },
    { host: 'combined.abuse.ch', name: 'Abuse.ch', category: 'Malware' },
    { host: 'dnsbl.dronebl.org', name: 'DroneBL', category: 'Malware' },
    { host: 'dnsbl.inps.de', name: 'INPS', category: 'Reputation' },
    { host: 'dnsbl.justspam.org', name: 'JustSpam', category: 'Reputation' },
    { host: 'dnsbl.manitu.net', name: 'Manitu', category: 'Reputation' },
    { host: 'dnsbl.spfbl.net', name: 'SPFBL', category: 'Reputation' },
    { host: 'drone.abuse.ch', name: 'Abuse.ch Drone', category: 'Malware' },
    { host: 'duinv.aupads.org', name: 'AUPADS', category: 'Reputation' },
    { host: 'dul.dnsbl.sorbs.net', name: 'SORBS DUL', category: 'Reputation' },
    { host: 'dyna.spamrats.com', name: 'SpamRats Dyna', category: 'Reputation' },
    { host: 'ips.backscatterer.org', name: 'Backscatterer', category: 'Reputation' },
    { host: 'ix.dnsbl.manitu.net', name: 'Manitu IX', category: 'Reputation' },
    { host: 'korea.services.net', name: 'Korea Services', category: 'Reputation' },
    { host: 'noptr.spamrats.com', name: 'SpamRats NoPTR', category: 'Reputation' },
    { host: 'ohps.dnsbl.net.au', name: 'OHPS', category: 'Reputation' },
    { host: 'omrs.dnsbl.net.au', name: 'OMRS', category: 'Reputation' },
    { host: 'orvedb.aupads.org', name: 'AUPADS ORVEDB', category: 'Reputation' },
    { host: 'osps.dnsbl.net.au', name: 'OSPS', category: 'Reputation' },
    { host: 'pbl.spamhaus.org', name: 'Spamhaus PBL', category: 'Critical' },
    { host: 'phishing.abuse.ch', name: 'Abuse.ch Phishing', category: 'Malware' },
    { host: 'proxy.bl.gweep.ca', name: 'Gweep Proxy', category: 'Reputation' },
    { host: 'rbl.interserver.net', name: 'InterServer', category: 'Reputation' },
    { host: 'relays.bl.gweep.ca', name: 'Gweep Relays', category: 'Reputation' },
    { host: 'relays.bl.kundenserver.de', name: 'Kundenserver', category: 'Reputation' }
];

export async function POST(request: Request) {
    try {
        const { ip } = await request.json();

        if (!ip) {
            return NextResponse.json({ error: 'IP address is required' }, { status: 400 });
        }

        const parts = ip.split('.');
        if (parts.length !== 4) {
            return NextResponse.json({ error: 'Invalid IPv4 address' }, { status: 400 });
        }
        const reversedIp = parts.reverse().join('.');

        const results = [];
        const batchSize = 25;

        // Use Google DNS-over-HTTPS to bypass local Port 53 blocks
        const checkRbl = async (rbl: any) => {
            const queryName = `${reversedIp}.${rbl.host}`;
            const dohUrl = `https://dns.google/resolve?name=${queryName}&type=A`;
            const start = Date.now();

            try {
                const response = await fetch(dohUrl, {
                    headers: { 'Accept': 'application/dns-json' },
                    next: { revalidate: 60 } // Cache for 60s
                });
                const data = await response.json();
                const latency = Date.now() - start;

                // Status 0 = NOERROR (Usually means Listed if Answer is present)
                // Status 3 = NXDOMAIN (Clean)

                if (data.Status === 3) {
                    return { ...rbl, status: 'clean', responseTime: latency };
                }

                if (data.Status === 0 && data.Answer && data.Answer.length > 0) {
                    // Get the A records (e.g. 127.0.0.2)
                    const codes = data.Answer.map((a: any) => a.data).join(', ');
                    return {
                        ...rbl,
                        status: 'listed',
                        responseTime: latency,
                        details: codes
                    };
                }

                // Any other status or empty answer usually means clean or ignored
                return { ...rbl, status: 'clean', responseTime: latency };

            } catch (err) {
                return { ...rbl, status: 'error', responseTime: Date.now() - start, details: 'DoH Failed' };
            }
        };

        for (let i = 0; i < RBL_DATABASE.length; i += batchSize) {
            const batch = RBL_DATABASE.slice(i, i + batchSize);
            const batchResults = await Promise.all(batch.map(checkRbl));
            results.push(...batchResults);
        }

        return NextResponse.json({ results });

    } catch (error: any) {
        console.error("Blacklist Check Error:", error);
        return NextResponse.json({ error: 'Failed to scan blacklists' }, { status: 500 });
    }
}
