import { NextResponse } from 'next/server';

// CDN Signatures Database
interface CDNProvider {
    id: string;
    name: string;
    headers: string[];
    domains: string[];
    orgs: RegExp[];
}

const PROVIDERS: CDNProvider[] = [
    {
        id: 'cloudflare',
        name: 'Cloudflare',
        headers: ['cf-ray', 'cf-cache-status', '__cfduid', 'server: cloudflare'],
        domains: ['cloudflare.com', 'cloudflare.net'],
        orgs: [/Cloudflare/i]
    },
    {
        id: 'akamai',
        name: 'Akamai',
        headers: ['x-akamai-transformed', 'x-akamai-request-id', 'server: akamaighost'],
        domains: ['akamai.net', 'akadns.net'],
        orgs: [/Akamai/i]
    },
    {
        id: 'fastly',
        name: 'Fastly',
        headers: ['x-fastly-request-id', 'fastly-restarts'],
        domains: ['fastly.net', 'fastlylb.net'],
        orgs: [/Fastly/i]
    },
    {
        id: 'cloudfront',
        name: 'AWS CloudFront',
        headers: ['x-amz-cf-id', 'via: cloudfront'],
        domains: ['cloudfront.net'],
        orgs: [/Amazon/i, /CloudFront/i]
    },
    {
        id: 'vercel',
        name: 'Vercel',
        headers: ['x-vercel-id', 'server: vercel'],
        domains: ['vercel-dns.com'],
        orgs: [/Vercel/i]
    },
    {
        id: 'netlify',
        name: 'Netlify',
        headers: ['x-nf-request-id', 'server: netlify'],
        domains: ['netlify.com'],
        orgs: [/Netlify/i]
    },
    {
        id: 'azure',
        name: 'Azure CDN',
        headers: [],
        domains: ['azure-dns.com', 'azure-dns.net'],
        orgs: [/Microsoft/i, /Azure/i]
    },
    {
        id: 'google',
        name: 'Google Cloud CDN',
        headers: ['via: 1.1 google'],
        domains: ['googledomains.com', 'cloud-dns.google'],
        orgs: [/Google/i]
    }
];

// Helper: Resolve DNS via DoH
async function resolveDoH(domain: string, type: 'A' | 'NS') {
    try {
        const res = await fetch(`https://dns.google/resolve?name=${domain}&type=${type}`, { cache: 'no-store' });
        const data = await res.json();
        return data.Answer || [];
    } catch {
        return [];
    }
}

// Helper: Get IP Info
async function getIpInfo(ip: string) {
    try {
        const res = await fetch(`http://ip-api.com/json/${ip}?fields=org,isp`, { signal: AbortSignal.timeout(3000) });
        if (!res.ok) return null;
        return await res.json();
    } catch {
        return null;
    }
}

export async function POST(request: Request) {
    try {
        const { url } = await request.json();
        if (!url) return NextResponse.json({ error: 'URL required' }, { status: 400 });

        // Normalize URL
        const target = url.startsWith('http') ? url : `https://${url}`;
        let hostname = '';
        try {
            hostname = new URL(target).hostname.toLowerCase();
        } catch {
            return NextResponse.json({ error: 'Invalid URL' }, { status: 400 });
        }

        // SSRF Prevention
        if (hostname === 'localhost' || hostname.startsWith('127.') || hostname.startsWith('192.168.') || hostname === '::1' || hostname.startsWith('10.')) {
            return NextResponse.json({ error: 'Restricted target' }, { status: 403 });
        }

        // Parallel Checks
        // 1. HTTP Headers Check
        const headerCheck = fetch(target, { method: 'HEAD', signal: AbortSignal.timeout(5000) })
            .then(res => {
                const headers: string[] = [];
                res.headers.forEach((v, k) => headers.push(`${k}: ${v}`));
                return headers;
            })
            .catch(() => [] as string[]);

        // 2. DNS NS Check
        const nsCheck = resolveDoH(hostname, 'NS');

        // 3. IP Check (Start with A record, then IP Info)
        const ipCheck = resolveDoH(hostname, 'A').then(async (answers: any[]) => {
            const ip = answers?.find((a: any) => a.type === 1)?.data;
            if (ip) {
                const info = await getIpInfo(ip);
                return { ip, org: info?.org || info?.isp || '' };
            }
            return { ip: '', org: '' };
        });

        const [headers, nsRecords, ipInfo] = await Promise.all([headerCheck, nsCheck, ipCheck]);

        let matchedProvider: CDNProvider | null = null;
        let detectedBy = 'None';
        let headersFound: string[] = [];

        // Analysis Loop
        for (const provider of PROVIDERS) {

            // Check Headers
            const matchingHeaders = provider.headers.filter(ph => {
                // Handle "key: value" vs just "key"
                if (ph.includes(':')) {
                    const [k, v] = ph.split(':').map(s => s.trim().toLowerCase());
                    return headers.some(h => {
                        const [hk, hv] = h.split(':').map(s => s.trim().toLowerCase());
                        return hk === k && hv.includes(v);
                    });
                }
                return headers.some(h => h.toLowerCase().startsWith(ph));
            });

            if (matchingHeaders.length > 0) {
                matchedProvider = provider;
                detectedBy = 'Headers';
                headersFound = headers.filter(h => matchingHeaders.some(mh => h.toLowerCase().includes(mh.split(':')[0])));
                break;
            }

            // Check NS
            if (nsRecords && nsRecords.some((ns: any) => provider.domains.some(d => ns.data.includes(d)))) {
                matchedProvider = provider;
                detectedBy = 'DNS';
                break;
            }

            // Check IP Org
            if (ipInfo.org && provider.orgs.some(regex => regex.test(ipInfo.org))) {
                matchedProvider = provider;
                detectedBy = 'IP';
                break;
            }
        }

        // Determine matching headers generic list if not already filled
        if (!matchingHeadersFilled(headersFound) && matchedProvider) {
            // Fill it? Actually logic above handles it for header detection.
            // If detected by IP/DNS, we might still want to see if any generic headers exist?
            // For simplicity, we just show identified headers if the detection was via Headers. 
            // If detected by IP, we might show IP.
        }

        function matchingHeadersFilled(list: string[]) { return list.length > 0; }

        const isWafActive = headers.some(h =>
            h.toLowerCase().includes('waf') ||
            h.toLowerCase().includes('firewall') ||
            (matchedProvider?.id === 'cloudflare' && headers.some(x => x.includes('cf-ray')))
        );

        const result = {
            name: matchedProvider ? matchedProvider.name : 'Direct / Unknown',
            providerId: matchedProvider ? matchedProvider.id : 'direct',
            ip: ipInfo.ip,
            organization: ipInfo.org || 'Direct-to-Origin',
            headersFound: headersFound.slice(0, 5), // Limit
            isWafActive,
            detectedBy: matchedProvider ? detectedBy : 'None'
        };

        return NextResponse.json(result);

    } catch (error: any) {
        return NextResponse.json({ error: error.message }, { status: 500 });
    }
}
