import { NextResponse } from 'next/server';

// Helper for A-record resolution via Google DoH
async function resolveIpDoH(hostname: string): Promise<string | null> {
    try {
        const controller = new AbortController();
        const timeout = setTimeout(() => controller.abort(), 3000); // 3s per IP max

        const res = await fetch(`https://dns.google/resolve?name=${hostname}&type=1`, {
            headers: { 'Accept': 'application/dns-json' },
            signal: controller.signal,
            next: { revalidate: 300 } // Cache for 5 mins
        });
        clearTimeout(timeout);

        if (!res.ok) return null;
        const data = await res.json();

        if (data.Answer) {
            // Find first A record (Type 1)
            const record = data.Answer.find((r: any) => r.type === 1);
            return record ? record.data : null;
        }
        return null;
    } catch (e) {
        return null;
    }
}

export async function POST(request: Request) {
    try {
        const { url } = await request.json();
        if (!url) return NextResponse.json({ error: 'Domain required' }, { status: 400 });

        const hostname = url.replace(/^https?:\/\//, '').replace(/\/$/, '');
        const unique = new Map<string, { subdomain: string, lastSeen: string, provider: string }>();

        // Helper to add results
        const addResult = (sub: string, source: string) => {
            let cleanName = sub.replace('*.', '').toLowerCase();
            // Remove leading dot if exists
            if (cleanName.startsWith('.')) cleanName = cleanName.substring(1);

            if (!cleanName.endsWith(hostname)) return;

            // Allow overwriting if newer or just ensure distinct
            if (!unique.has(cleanName)) {
                unique.set(cleanName, {
                    subdomain: cleanName,
                    lastSeen: new Date().toISOString(),
                    provider: source
                });
            }
        };

        // --- STRATEGY 1: CRT.SH (Primary) ---
        try {
            const crtController = new AbortController();
            const crtTimeout = setTimeout(() => crtController.abort(), 8000); // 8s Timeout

            const crtResponse = await fetch(`https://crt.sh/?q=%.${hostname}&output=json`, {
                signal: crtController.signal,
                headers: { 'User-Agent': 'Mozilla/5.0 (compatible; DNSTool/1.0)' }
            });
            clearTimeout(crtTimeout);

            if (crtResponse.ok) {
                const data = await crtResponse.json();
                data.forEach((entry: any) => {
                    const names = entry.name_value.split('\n');
                    names.forEach((name: string) => {
                        // CRT.sh often returns wildecards like *.example.com or just example.com
                        addResult(name, 'CT Log');
                    });
                });
            }
        } catch (e) {
            console.warn("CRT.sh failed, trying fallback...", e);
        }

        // --- STRATEGY 2: HACKERTARGET (Fallback if CRT failed or empty) ---
        if (unique.size === 0) {
            try {
                const htController = new AbortController();
                const htTimeout = setTimeout(() => htController.abort(), 5000);

                const htResponse = await fetch(`https://api.hackertarget.com/hostsearch/?q=${hostname}`, {
                    signal: htController.signal
                });
                clearTimeout(htTimeout);

                if (htResponse.ok) {
                    const text = await htResponse.text();
                    const lines = text.split('\n');
                    lines.forEach(line => {
                        // Format: hostname,ip
                        const [host] = line.split(',');
                        if (host && host.includes(hostname)) {
                            addResult(host, 'DNS Scan');
                        }
                    });
                }
            } catch (e) {
                console.warn("HackerTarget failed...", e);
            }
        }

        if (unique.size === 0) {
            throw new Error("No subdomains found. Providers may be busy or domain has no visible subdomains.");
        }

        const sorted = Array.from(unique.values())
            .sort((a, b) => new Date(b.lastSeen).getTime() - new Date(a.lastSeen).getTime());

        // 3. Resolve IPs for top 20 using DoH
        const top20 = sorted.slice(0, 20);
        const resolutionPromises = top20.map(async (item) => {
            const ip = await resolveIpDoH(item.subdomain);
            return {
                ...item,
                ip: ip,
                isReachable: !!ip
            };
        });

        const resolvedTop = await Promise.all(resolutionPromises);

        // Merge back
        const finalResults = [
            ...resolvedTop,
            ...sorted.slice(20).map(item => ({ ...item, ip: null, isReachable: undefined }))
        ];

        return NextResponse.json({
            subdomains: finalResults,
            total: finalResults.length,
            uniqueIps: new Set(resolvedTop.map(r => r.ip).filter(Boolean)).size
        });

    } catch (error: any) {
        return NextResponse.json({ error: error.message || "Discovery Failed" }, { status: 500 });
    }
}
