import { NextResponse } from 'next/server';

export async function POST(request: Request) {
    try {
        const { domain, selector } = await request.json();

        if (!domain || !selector) {
            return NextResponse.json({ error: 'Domain and Selector are required' }, { status: 400 });
        }

        const cleanDomain = domain.replace(/^https?:\/\//, '').replace(/\/$/, '');
        const target = `${selector}._domainkey.${cleanDomain}`;

        // Fetch via Google DoH
        const res = await fetch(`https://dns.google/resolve?name=${target}&type=TXT`, {
            headers: { 'Accept': 'application/dns-json' },
            next: { revalidate: 60 }
        });

        if (!res.ok) throw new Error("DNS Query failed");

        const data = await res.json();

        if (!data.Answer || data.Answer.length === 0) {
            return NextResponse.json({ found: false });
        }

        // Parse TXT
        // data.Answer[0].data is usually "v=DKIM1; k=rsa; p=..." (quoted)
        // clean quotes
        const raw = data.Answer[0].data.replace(/^"|"$/g, '').replace(/\\"/g, '"');

        const parseTag = (tag: string) => {
            const match = raw.match(new RegExp(`${tag}=([^;]+)`));
            return match ? match[1].trim() : '';
        };

        const version = parseTag('v');
        const keyType = parseTag('k') || 'rsa'; // Default is RSA
        const publicKey = parseTag('p');
        const flags = parseTag('t');

        // Estimate Key Length
        let keyLength = 0;
        if (publicKey) {
            try {
                // Decode base64
                const binString = atob(publicKey);
                // RSA keys in DKIM are DER encoded SubjectPublicKeyInfo usually.
                // The bit length is simpler to approximate via byte length * 8
                // For 1024 bit key, DER length is approx 162 bytes.
                // For 2048 bit key, DER length is approx 294 bytes.
                // Let's just use byte length * 8 as a rough upper bound, or closer logic.
                // Actually, key length usually refers to the modulus size.
                // Parsing ASN.1 in raw JS without libs is pain.
                // Heuristic: 
                // 1024 bit ~ 160-170 bytes
                // 2048 bit ~ 270-295 bytes
                // 4096 bit ~ 550+ bytes
                const len = binString.length;
                if (len > 500) keyLength = 4096;
                else if (len > 250) keyLength = 2048;
                else if (len > 120) keyLength = 1024;
                else keyLength = len * 8; // fallback
            } catch (e) {
                // invalid base64
            }
        }

        return NextResponse.json({
            found: true,
            record: {
                raw,
                version,
                keyType: keyType.toUpperCase(),
                publicKey,
                keyLength,
                flags,
                isKeyValid: !!publicKey
            }
        });

    } catch (error: any) {
        return NextResponse.json({ error: error.message }, { status: 500 });
    }
}
