import { NextResponse } from 'next/server';

// Reuse DoH Pattern for consistency & reliability
async function resolveTxtDoH(domain: string) {
    try {
        const res = await fetch(`https://dns.google/resolve?name=${domain}&type=TXT`, { cache: 'no-store' });
        if (!res.ok) throw new Error(`DoH Error: ${res.statusText}`);
        const data = await res.json();

        if (data.Status !== 0 || !data.Answer) return []; // No records

        // Google DoH returns TXT data in quotes ("v=DMARC1..."), sometimes disjointed
        return data.Answer.map((ans: any) => {
            // Strip quotes
            let txt = ans.data;
            if (txt.startsWith('"') && txt.endsWith('"')) {
                txt = txt.slice(1, -1);
            }
            // Some responses like "v=DMARC1;" " p=reject" might need joining if split by DNS chunks
            // But Google usually handles this well in JSON.
            return txt;
        });
    } catch (e) {
        console.error("TxtDoH Error:", e);
        return [];
    }
}

export async function POST(request: Request) {
    try {
        const { domain } = await request.json();

        if (!domain) {
            return NextResponse.json({ error: 'Domain is required' }, { status: 400 });
        }

        const dmarcDomain = `_dmarc.${domain}`;

        // 1. Resolve TXT Records via DoH
        const txtRecords = await resolveTxtDoH(dmarcDomain);

        if (!txtRecords || txtRecords.length === 0) {
            return NextResponse.json({ error: 'No DMARC record found at _dmarc.' + domain }, { status: 404 });
        }

        // 2. Find the DMARC record (must start with v=DMARC1)
        // Case-insensitive check for the tag, but usually it's upper.
        const dmarcRecord = txtRecords.find((r: string) => r.trim().toUpperCase().startsWith('V=DMARC1'));

        if (!dmarcRecord) {
            return NextResponse.json({ error: 'No valid DMARC record found (missing v=DMARC1).' }, { status: 404 });
        }

        return NextResponse.json({
            domain,
            record: dmarcRecord
        });

    } catch (error: any) {
        console.error("DMARC Lookup Error:", error);
        return NextResponse.json({ error: `Lookup failed: ${error.message}` }, { status: 500 });
    }
}
