import { NextResponse } from 'next/server';

// Helper: Resolve via Google DoH
async function resolveDoH(domain: string): Promise<string[]> {
    try {
        const res = await fetch(`https://dns.google/resolve?name=${domain}&type=TXT`, { cache: 'no-store' });
        if (!res.ok) return [];
        const data = await res.json();
        if (data.Status !== 0 || !data.Answer) return [];

        // Extract TXT data
        return data.Answer.map((a: any) => a.data ? a.data.replace(/^"|"$/g, '') : '');
    } catch (e) {
        return [];
    }
}

interface ParsedMech {
    prefix: string;
    type: string;
    value: string;
    isLookup: boolean;
    description?: string;
}

// Mechanisms that cause a DNS lookup
const LOOKUP_MECHANISMS = ['include', 'a', 'mx', 'ptr', 'exists', 'redirect'];

export async function POST(request: Request) {
    try {
        const { domain } = await request.json();
        if (!domain) return NextResponse.json({ error: 'Domain required' }, { status: 400 });

        // 1. Fetch Root Record
        const txtRecords = await resolveDoH(domain);
        const spfRaw = txtRecords.find(r => r.includes('v=spf1'));

        if (!spfRaw) {
            return NextResponse.json({ error: 'No SPF record found', found: false }, { status: 404 });
        }

        // 2. Parse Mechanisms
        const parts = spfRaw.split(' ').slice(1); // Remove v=spf1
        const mechanisms: ParsedMech[] = [];
        let lookupCount = 0;
        let errors: string[] = [];

        // Helper to check recursive lookups
        // Note: For this MVP, we will count the top-level lookups and 1 level deep for 'includes' 
        // to approximate the count without full 100% compliant deep tree traversal which can be slow.
        // Or we can try a slightly deeper async crawl.

        for (const part of parts) {
            if (!part.trim()) continue;

            // Regex Parse: ([qualifier])(mechanism)[:=](value)
            // Simple approach:
            const match = part.match(/^([+\-~?])?([a-zA-Z0-9]+)([:=](.+))?$/);

            if (match) {
                const prefix = match[1] || '+';
                const type = match[2].toLowerCase();
                const value = match[4] || ''; // Value after : or =

                const isLookup = LOOKUP_MECHANISMS.includes(type);
                if (isLookup) lookupCount++;

                mechanisms.push({
                    prefix,
                    type,
                    value,
                    isLookup
                });

                // Deep Lookup for Include count
                if (type === 'include' && value) {
                    // We need to fetch that record too to update the REAL lookup count?
                    // The RFC limit counts elements that CAUSE a look up.
                    // 'include:generic.ptr' counts as 1. 
                    // BUT, the nested record's lookups ALSO count towards the total limit of the ORIGINAL validation.
                    // So yes, we MUST fetch included records to sum their lookups.

                    try {
                        const includedTxts = await resolveDoH(value);
                        const includedSpf = includedTxts.find(r => r.includes('v=spf1'));
                        if (includedSpf) {
                            const subParts = includedSpf.split(' ');
                            for (const sub of subParts) {
                                const subType = sub.split(/[:=]/)[0].replace(/^[+\-~?]/, '').toLowerCase();
                                if (LOOKUP_MECHANISMS.includes(subType)) {
                                    lookupCount++;
                                }
                            }
                        }
                    } catch (e) {
                        errors.push(`Failed to fetch included record: ${value}`);
                    }
                }

            } else {
                // Unknown format, maybe just a modifier
                mechanisms.push({ prefix: '', type: 'unknown', value: part, isLookup: false });
            }
        }

        // Validate
        if (lookupCount > 10) errors.push("DNS Lookup Limit Exceeded (>10).");
        if (spfRaw.length > 450) errors.push("Record may exceed UDP packet limits (>450b)."); // Heuristic

        return NextResponse.json({
            raw: spfRaw,
            mechanisms,
            lookupCount,
            isValid: errors.length === 0,
            errors
        });

    } catch (error: any) {
        return NextResponse.json({ error: error.message }, { status: 500 });
    }
}
