import { NextResponse } from 'next/server';

export async function GET(request: Request) {
    try {
        const { searchParams } = new URL(request.url);
        let ip = searchParams.get('ip') || '';

        // Auto-detect IP if not provided
        if (!ip) {
            const forwarded = request.headers.get('x-forwarded-for');
            ip = forwarded ? forwarded.split(',')[0] : '127.0.0.1';
        }

        // Handle Localhost / Private IPs immediately to prevent API errors
        if (ip === '::1' || ip === '127.0.0.1' || ip.startsWith('192.168.') || ip.startsWith('10.') || ip.startsWith('172.')) {
            return NextResponse.json({
                ...mockLocalhostData,
                ip: ip // Use the actual local IP
            });
        }

        const fields = "status,message,country,countryCode,regionName,city,lat,lon,timezone,isp,org,as,mobile,proxy,hosting,query";
        const apiUrl = `http://ip-api.com/json/${ip}?fields=${fields}`;

        const res = await fetch(apiUrl);
        if (!res.ok) {
            return NextResponse.json({ error: `API Error: ${res.statusText}` }, { status: res.status });
        }
        const data = await res.json();

        if (data.status === 'fail') {
            // Handle private range or reserved range returned by API
            if (data.message === 'private range' || data.message === 'reserved range') {
                return NextResponse.json({
                    ...mockLocalhostData,
                    ip: ip
                });
            }
            return NextResponse.json({ error: data.message }, { status: 400 });
        }

        // Simulating Fraud/Risk Score based on available signals
        let riskScore = 0;
        if (data.hosting) riskScore += 50;
        if (data.proxy) riskScore += 30;
        if (data.mobile) riskScore -= 10;

        const normalized = {
            ip: data.query,
            city: data.city,
            region: data.regionName,
            country: data.country,
            countryCode: data.countryCode,
            lat: data.lat,
            lon: data.lon,
            asn: data.as ? data.as.split(' ')[0] : 'N/A',
            asnName: data.org || data.isp,
            asnType: data.hosting ? "Hosting" : data.mobile ? "Mobile" : "ISP",
            isp: data.isp,
            timezone: data.timezone,
            proxy: data.proxy || false,
            vpn: data.proxy || false,
            tor: false,
            hosting: data.hosting || false,
            fraudScore: Math.max(0, Math.min(100, riskScore))
        };

        return NextResponse.json(normalized);

    } catch (error: any) {
        console.error("IP Info Error:", error);
        return NextResponse.json({ error: 'Failed to fetch IP info' }, { status: 500 });
    }
}

const mockLocalhostData = {
    ip: '127.0.0.1',
    city: 'Local Network',
    region: 'Private',
    country: 'Local System',
    countryCode: 'US', // Fallback for flag
    lat: 0,
    lon: 0,
    asn: 'AS0000',
    asnName: 'Loopback Interface',
    asnType: 'Private',
    isp: 'Local System',
    timezone: 'UTC',
    proxy: false,
    vpn: false,
    tor: false,
    hosting: false,
    fraudScore: 0
};
