import { NextResponse } from 'next/server';
import { spawn } from 'node:child_process';
import os from 'node:os';

export const runtime = 'nodejs'; // Required for child_process

export async function POST(request: Request) {
    try {
        const { host, count = 4 } = await request.json();

        if (!host) {
            return NextResponse.json({ error: 'Host is required' }, { status: 400 });
        }

        // 1. Strict Sanitization
        // Allow alphanumeric, dots, hyphens. Deny spaces, semi-colons, etc.
        const isValidHost = /^[a-zA-Z0-9.-]+$/.test(host);
        if (!isValidHost) {
            return NextResponse.json({ error: 'Invalid hostname format' }, { status: 400 });
        }

        // Limit count
        const safeCount = Math.min(Math.max(1, Number(count)), 50); // Max 50 for safety

        // 2. Determine Command
        const isWindows = os.platform() === 'win32';
        const command = 'ping';
        const args = isWindows
            ? ['-n', safeCount.toString(), host]
            : ['-c', safeCount.toString(), host];

        // 3. Create Stream
        const encoder = new TextEncoder();
        const stream = new ReadableStream({
            start(controller) {
                const child = spawn(command, args);

                child.stdout.on('data', (chunk) => {
                    const text = chunk.toString();
                    controller.enqueue(encoder.encode(text));
                });

                child.stderr.on('data', (chunk) => {
                    // Pings often send "errors" to stdout mostly, but capture stderr just in case
                    const text = chunk.toString();
                    controller.enqueue(encoder.encode(text));
                });

                child.on('error', (err) => {
                    controller.enqueue(encoder.encode(`\nError: ${err.message}\n`));
                    controller.close();
                });

                child.on('close', (code) => {
                    controller.close();
                });
            }
        });

        return new Response(stream, {
            headers: {
                'Content-Type': 'text/plain; charset=utf-8',
                'Transfer-Encoding': 'chunked',
                'X-Content-Type-Options': 'nosniff',
            },
        });

    } catch (error: any) {
        return NextResponse.json({ error: error.message }, { status: 500 });
    }
}
