import { NextResponse } from 'next/server';

interface RedirectHop {
    url: string;
    status: number;
    statusText: string;
    timing: number;
    location: string | null;
    server?: string;
}

const MAX_HOPS = 15;

const USER_AGENTS = {
    desktop: 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
    mobile: 'Mozilla/5.0 (iPhone; CPU iPhone OS 17_0 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/17.0 Mobile/15E148 Safari/604.1'
};

export async function POST(request: Request) {
    try {
        const { url, device = 'desktop' } = await request.json();

        if (!url) return NextResponse.json({ error: 'URL is required' }, { status: 400 });

        let currentUrl = url.startsWith('http') ? url : `https://${url}`;
        const hops: RedirectHop[] = [];
        const visited = new Set<string>();
        let isLoop = false;

        const userAgent = device === 'mobile' ? USER_AGENTS.mobile : USER_AGENTS.desktop;

        for (let i = 0; i < MAX_HOPS; i++) {
            // Check Loop
            if (visited.has(currentUrl)) {
                isLoop = true;
                break;
            }
            visited.add(currentUrl);

            const start = performance.now();
            let response: Response;

            try {
                response = await fetch(currentUrl, {
                    method: 'GET',
                    redirect: 'manual', // MANUAL FOLLOW
                    headers: { 'User-Agent': userAgent }
                });
            } catch (err: any) {
                // Network error (e.g., DNS failed)
                hops.push({
                    url: currentUrl,
                    status: 0,
                    statusText: err.message || 'Network Error',
                    timing: Math.round(performance.now() - start),
                    location: null
                });
                break;
            }

            const timing = Math.round(performance.now() - start);
            const status = response.status;
            const location = response.headers.get('location');

            // Resolve relative location
            let nextUrl = location;
            if (location && !location.startsWith('http')) {
                try {
                    nextUrl = new URL(location, currentUrl).toString();
                } catch {
                    // Keep as is if invalid
                }
            }

            hops.push({
                url: currentUrl,
                status,
                statusText: response.statusText,
                timing,
                location: nextUrl,
                server: response.headers.get('server') || undefined
            });

            if (status >= 300 && status < 400 && nextUrl) {
                currentUrl = nextUrl;
            } else {
                // Not a redirect or no location, stop
                break;
            }
        }

        return NextResponse.json({ hops, isLoop });

    } catch (error: any) {
        return NextResponse.json({ error: error.message }, { status: 500 });
    }
}
