"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { ShieldAlert, CheckCircle, ExternalLink, Activity, AlertTriangle, Bug, Search, EyeOff } from "lucide-react";

export interface BlacklistResult {
    host: string;
    name: string;
    category: 'Critical' | 'Reputation' | 'Malware' | 'General';
    status: 'listed' | 'clean' | 'error' | 'pending';
    responseTime: number;
    details?: string;
}

interface BlacklistDashboardProps {
    results: BlacklistResult[];
    scannedCount: number;
    totalLists: number;
    ip: string;
}

export const BlacklistDashboard: React.FC<BlacklistDashboardProps> = ({ results, scannedCount, totalLists, ip }) => {

    const [activeTab, setActiveTab] = useState<'All' | 'Critical' | 'Reputation' | 'Malware'>('All');
    const [filterText, setFilterText] = useState("");

    // Filter & Sort Logic
    // 1. Filter by Tab/Search
    // 2. Sort: Listed -> Error -> Clean (alphabetical)
    const processedResults = results.filter(r => {
        const matchesTab = activeTab === 'All' ? true : r.category === activeTab;
        const matchesText = r.name.toLowerCase().includes(filterText.toLowerCase()) || r.host.toLowerCase().includes(filterText.toLowerCase());
        return matchesTab && matchesText;
    }).sort((a, b) => {
        // Priority: Listed > Error > Clean
        const score = (status: string) => {
            if (status === 'listed') return 3;
            if (status === 'error') return 2;
            return 1;
        };
        if (score(b.status) !== score(a.status)) {
            return score(b.status) - score(a.status);
        }
        return a.name.localeCompare(b.name);
    });

    const getIcon = (category: string) => {
        switch (category) {
            case 'Critical': return <AlertTriangle className="w-4 h-4 text-orange-500" />;
            case 'Malware': return <Bug className="w-4 h-4 text-red-500" />;
            default: return <Activity className="w-4 h-4 text-blue-500" />;
        }
    };

    return (
        <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">

            {/* Main Result Table */}
            <div className="lg:col-span-3 space-y-6">

                {/* Control Bar */}
                <div className="flex flex-col sm:flex-row items-center justify-between gap-4 p-1 rounded-xl bg-slate-100/50 border border-slate-200">
                    <div className="flex items-center gap-1 bg-white rounded-lg p-1 shadow-sm border border-slate-200">
                        {['All', 'Critical', 'Reputation', 'Malware'].map((tab) => (
                            <button
                                key={tab}
                                onClick={() => setActiveTab(tab as any)}
                                className={`px-4 py-2 text-xs font-bold rounded-md transition-all
                                    ${activeTab === tab
                                        ? "bg-slate-900 text-white shadow-md"
                                        : "text-slate-500 hover:text-slate-900 hover:bg-slate-50"}
                                `}
                            >
                                {tab}
                            </button>
                        ))}
                    </div>

                    <div className="relative w-full sm:w-64 group">
                        <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                        <input
                            type="text"
                            placeholder="Filter lists..."
                            value={filterText}
                            onChange={(e) => setFilterText(e.target.value)}
                            className="w-full h-10 pl-9 pr-4 bg-white border border-slate-200 rounded-lg text-sm font-medium focus:outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500 transition-all"
                        />
                    </div>
                </div>

                {/* Table - Designed to match HetrixTools Layout */}
                <div className="bg-[#1E293B] border border-slate-800 rounded-xl overflow-hidden shadow-xl ring-1 ring-white/10">
                    <div className="overflow-x-auto">
                        <table className="w-full text-left text-sm">
                            <thead className="bg-[#0F172A] text-slate-400 uppercase text-xs font-bold tracking-wider">
                                <tr>
                                    <th className="px-6 py-4">RBL Database</th>
                                    <th className="px-6 py-4 text-right w-24">Action</th>
                                    <th className="px-6 py-4 text-center w-32">Status</th>
                                    <th className="px-6 py-4 text-right w-32">Resolution</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y divide-slate-700/50">
                                {processedResults.length === 0 && (
                                    <tr>
                                        <td colSpan={4} className="px-6 py-12 text-center text-slate-500 italic bg-[#1E293B]">
                                            No matching databases found.
                                        </td>
                                    </tr>
                                )}
                                {processedResults.map((r, i) => (
                                    <motion.tr
                                        key={r.host}
                                        layout
                                        initial={{ opacity: 0 }} animate={{ opacity: 1 }}
                                        className={`group transition-colors border-l-4
                                            ${r.status === 'listed'
                                                ? "bg-[#450a0a]/90 hover:bg-[#500a0a] border-l-red-500"
                                                : "bg-[#1E293B] hover:bg-[#334155] border-l-transparent"
                                            }
                                        `}
                                    >
                                        <td className="px-6 py-4">
                                            <div className="flex items-center gap-3">
                                                <span className={`${r.status === 'listed' ? "text-red-100 font-bold" : "text-slate-300 font-medium"}`}>
                                                    {r.host}
                                                </span>
                                            </div>
                                        </td>

                                        {/* Ignore Action */}
                                        <td className="px-6 py-4 text-right">
                                            <button className="text-xs font-bold text-slate-500 hover:text-slate-300 transition-colors flex items-center justify-end gap-1">
                                                Ignore
                                            </button>
                                        </td>

                                        {/* Status Badge */}
                                        <td className="px-6 py-4 text-center">
                                            {r.status === 'listed' ? (
                                                <span className="inline-block px-3 py-1 bg-[#7f1d1d] text-red-100 border border-red-800 rounded text-xs font-bold w-full uppercase tracking-wider">
                                                    Listed
                                                </span>
                                            ) : r.status === 'error' ? (
                                                <span className="inline-block px-3 py-1 bg-orange-900/50 text-orange-200 border border-orange-800 rounded text-xs font-bold w-full">
                                                    Timeout
                                                </span>
                                            ) : (
                                                <span className="inline-block px-3 py-1 bg-[#064e3b] text-emerald-100 border border-emerald-800 rounded text-xs font-bold w-full uppercase tracking-wider">
                                                    Not Listed
                                                </span>
                                            )}
                                        </td>

                                        {/* Delist Action */}
                                        <td className="px-6 py-4 text-right">
                                            {r.status === 'listed' ? (
                                                <a
                                                    href={`https://google.com/search?q=delist+${ip}+from+${r.name}`}
                                                    target="_blank" rel="noopener noreferrer"
                                                    className="inline-block px-4 py-1.5 bg-[#7f1d1d] hover:bg-[#991b1b] text-red-100 border border-red-800 rounded text-xs font-bold w-full text-center transition-colors"
                                                >
                                                    Delist
                                                </a>
                                            ) : (
                                                <div className="h-7 w-full" />
                                            )}
                                        </td>
                                    </motion.tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            {/* Side Panel: Live Logs */}
            <div className="lg:col-span-1 space-y-6">

                {/* Visualizer */}
                <div className="bg-slate-900 text-slate-300 rounded-3xl p-5 shadow-2xl shadow-slate-900/20 border border-slate-800 font-mono text-xs overflow-hidden relative">
                    <div className="absolute top-0 inset-x-0 h-px bg-gradient-to-r from-transparent via-blue-500 to-transparent opacity-50" />

                    <div className="flex items-center justify-between text-slate-500 mb-4 text-[10px] uppercase font-bold tracking-wider">
                        <span className="flex items-center gap-2"><Activity className="w-3 h-3 text-blue-500" /> Live Audit Stream</span>
                        <div className="w-2 h-2 rounded-full bg-green-500 animate-pulse" />
                    </div>

                    <div className="h-[400px] overflow-y-auto space-y-3 scrollbar-hide mask-fade-bottom">
                        {results.slice().reverse().map((r, i) => (
                            <div key={i} className="flex gap-3 items-center group">
                                <span className="text-[10px] text-slate-600">{i + 1}</span>
                                <div className={`w-1.5 h-1.5 rounded-full shrink-0 ${r.status === 'listed' ? 'bg-red-500 shadow-[0_0_8px_rgba(239,68,68,0.6)]' : 'bg-emerald-500/50'}`} />
                                <div className="flex-1 min-w-0">
                                    <div className="truncate text-slate-400 group-hover:text-slate-200 transition-colors">{r.name}</div>
                                </div>
                                <span className={`text-[10px] font-bold ${r.status === 'listed' ? 'text-red-400' : 'text-slate-700'}`}>
                                    {r.status === 'listed' ? 'FAIL' : 'OK'}
                                </span>
                            </div>
                        ))}
                        {scannedCount < totalLists && (
                            <div className="text-blue-500 opacity-50 animate-pulse pt-2">Processing next batch...</div>
                        )}
                        {scannedCount === 0 && <span className="italic opacity-30">Waiting for target...</span>}
                    </div>
                </div>
            </div>

        </div>
    );
};
