"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Shield, Loader2, Fingerprint, Search, Key, HelpCircle, ChevronDown } from "lucide-react";

import { DKIMKeyDetails, DKIMRecord } from "@/components/dkim-tester/DKIMKeyDetails";

const COMMON_SELECTORS = [
    { value: "default", label: "default (Standard)" },
    { value: "google", label: "google (G Suite)" },
    { value: "k1", label: "k1 (Exchange/O365)" },
    { value: "s1", label: "s1 (Generic)" },
    { value: "selector1", label: "selector1 (Generic)" },
    { value: "mail", label: "mail" },
    { value: "mn", label: "mn (Mailgun)" },
    { value: "smtp", label: "smtp" },
    { value: "mandrill", label: "mandrill (Mailchimp)" },
    { value: "20230601", label: "20230601 (Google/Recent)" },
];

export default function DKIMClient() {
    const [domain, setDomain] = useState("");
    const [selector, setSelector] = useState("default");
    const [customSelector, setCustomSelector] = useState("");
    const [isCustom, setIsCustom] = useState(false);

    const [loading, setLoading] = useState(false);
    const [result, setResult] = useState<DKIMRecord | null>(null);
    const [notFound, setNotFound] = useState(false);
    const [error, setError] = useState<string | null>(null);

    const handleInspect = async (e?: React.FormEvent) => {
        if (e) e.preventDefault();

        const activeSelector = isCustom ? customSelector : selector;
        if (!domain || !activeSelector) return;

        setLoading(true);
        setError(null);
        setResult(null);
        setNotFound(false);

        try {
            const response = await fetch('/api/dns/dkim', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    domain: domain.trim(),
                    selector: activeSelector.trim()
                }),
            });

            const data = await response.json();

            if (!response.ok) throw new Error(data.error || "Inspection failed");

            if (!data.found) {
                setNotFound(true);
            } else {
                setResult(data.record);
            }

        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    const handleSelectorChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
        const val = e.target.value;
        if (val === 'other') {
            setIsCustom(true);
            setSelector('other');
        } else {
            setIsCustom(false);
            setSelector(val);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Fingerprint className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                DKIM <span className="text-blue-600">Inspector</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Retrieve and validate DKIM public keys to ensure your email digital signatures are cryptographically sound.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Search Box */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <form onSubmit={handleInspect} className="flex flex-col md:flex-row gap-4 items-start md:items-end">
                        <div className="flex-1 w-full space-y-1">
                            <label className="text-xs font-bold text-slate-500 uppercase tracking-wider ml-1">Domain</label>
                            <div className="relative group">
                                <Key className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                                <input
                                    type="text"
                                    placeholder="example.com"
                                    value={domain}
                                    onChange={(e) => setDomain(e.target.value)}
                                    className="w-full h-12 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                                />
                            </div>
                        </div>

                        <div className="flex-1 w-full space-y-1">
                            <label className="text-xs font-bold text-slate-500 uppercase tracking-wider ml-1 flex items-center gap-1">
                                Selector
                                <span className="text-slate-300" title="Find this in email headers (s=...)"><HelpCircle className="w-3 h-3 cursor-help" /></span>
                            </label>

                            <div className="flex flex-col gap-2">
                                {/* Dropdown */}
                                <div className="relative">
                                    <Fingerprint className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 pointer-events-none" />
                                    <select
                                        value={selector}
                                        onChange={handleSelectorChange}
                                        className="w-full h-12 pl-12 pr-10 bg-slate-50 border border-slate-200 rounded-xl font-medium text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all appearance-none cursor-pointer"
                                    >
                                        {COMMON_SELECTORS.map(s => (
                                            <option key={s.value} value={s.value}>{s.label}</option>
                                        ))}
                                        <option disabled>──────────</option>
                                        <option value="other">Other (Custom)...</option>
                                    </select>
                                    <ChevronDown className="absolute right-4 top-1/2 -translate-y-1/2 w-4 h-4 text-slate-400 pointer-events-none" />
                                </div>

                                {/* Custom Input Animation */}
                                <AnimatePresence>
                                    {isCustom && (
                                        <motion.div
                                            initial={{ opacity: 0, height: 0, marginTop: 0 }}
                                            animate={{ opacity: 1, height: 'auto', marginTop: 8 }}
                                            exit={{ opacity: 0, height: 0, marginTop: 0 }}
                                            className="overflow-hidden"
                                        >
                                            <div className="relative">
                                                <input
                                                    type="text"
                                                    placeholder="Enter custom selector..."
                                                    value={customSelector}
                                                    onChange={(e) => setCustomSelector(e.target.value)}
                                                    className="w-full h-12 pl-4 pr-4 bg-blue-50 border border-blue-200 rounded-xl font-mono text-blue-900 placeholder-blue-300 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                                                    autoFocus
                                                />
                                            </div>
                                        </motion.div>
                                    )}
                                </AnimatePresence>
                            </div>
                        </div>

                        <button
                            type="submit"
                            disabled={loading || !domain || (isCustom && !customSelector)}
                            className={`h-12 px-8 rounded-xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95 shrink-0 mb-[1px]
                                ${loading ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                            `}
                        >
                            {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : <Search className="w-5 h-5 fill-current" />}
                            {loading ? "Looking up..." : "Inspect Key"}
                        </button>
                    </form>
                </div>

                {loading && (
                    <div className="flex flex-col items-center justify-center py-20 text-center space-y-4">
                        <Loader2 className="w-10 h-10 text-blue-600 animate-spin" />
                        <p className="text-slate-500 font-medium animate-pulse">Querying DNS Records...</p>
                    </div>
                )}

                {notFound && !loading && (
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        className="bg-red-50 border border-red-100 rounded-3xl p-8 text-center"
                    >
                        <div className="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
                            <Fingerprint className="w-8 h-8 text-red-500" />
                        </div>
                        <h3 className="text-xl font-black text-red-900 mb-2">DKIM Record Not Found</h3>
                        <p className="text-red-800/70 max-w-md mx-auto">
                            We could not find a DKIM record at <span className="font-mono font-bold bg-white/50 px-1 rounded">{isCustom ? customSelector : selector}._domainkey.{domain}</span>.
                            Please check your selector and try again.
                        </p>
                    </motion.div>
                )}

                {result && (
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                    >
                        <div className="flex items-center gap-4 mb-6">
                            <div className="p-3 bg-emerald-100 rounded-xl">
                                <Shield className="w-6 h-6 text-emerald-600" />
                            </div>
                            <div>
                                <h2 className="text-lg font-black text-slate-900">DKIM Record Found</h2>
                                <p className="text-slate-500 text-sm">Valid public key published for this selector.</p>
                            </div>
                        </div>

                        <DKIMKeyDetails data={result} />
                    </motion.div>
                )}

                {/* Error State */}
                {error && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        className="bg-red-50 text-red-700 p-4 rounded-2xl flex items-center gap-3 border border-red-100"
                    >
                        <Fingerprint className="w-5 h-5 flex-shrink-0" />
                        <span className="font-medium">{error}</span>
                    </motion.div>
                )}

            </main>


        </div>
    );
}
