"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Shield, Search, Loader2, Globe, AlertCircle, FileText, BarChart3, Lock } from "lucide-react";

import { DMARCDeconstructor } from "@/components/dmarc-analyzer/DMARCDeconstructor";
import { DMARCSuggestion } from "@/components/dmarc-analyzer/DMARCSuggestion";

export default function DMARCClient() {
    const [domain, setDomain] = useState("");
    const [loading, setLoading] = useState(false);
    const [record, setRecord] = useState<string | null>(null);
    const [error, setError] = useState<string | null>(null);

    const handleSearch = async (e?: React.FormEvent) => {
        if (e) e.preventDefault();
        if (!domain) return;

        setLoading(true);
        setError(null);
        setRecord(null);

        try {
            const response = await fetch('/api/dns/dmarc', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ domain: domain.trim() }),
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || "Lookup failed");
            }

            setRecord(data.record);
        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    // Helper to calculate simple score/stats
    const getStats = (rec: string) => {
        const policy = rec.match(/p=(\w+)/)?.[1] || 'none';
        const hasRua = rec.includes('rua=');

        let score = 50; // Base score
        if (policy === 'reject') score += 40;
        else if (policy === 'quarantine') score += 20;

        if (hasRua) score += 10;

        // Cap at 100
        return { policy, hasRua, score: Math.min(100, score) };
    };

    const stats = record ? getStats(record) : null;

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Shield className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                DMARC Record <span className="text-blue-600">Intelligence</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Audit your domain’s DMARC policy, alignment modes, and reporting status to prevent email spoofing.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Search Box */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <form onSubmit={handleSearch} className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1 relative group">
                            <Globe className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                            <input
                                type="text"
                                placeholder="Enter Domain (e.g., google.com)"
                                value={domain}
                                onChange={(e) => setDomain(e.target.value)}
                                className="w-full h-14 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-2xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                            />
                        </div>

                        <button
                            type="submit"
                            disabled={loading || !domain}
                            className={`h-14 px-8 rounded-2xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95
                                ${loading ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                            `}
                        >
                            {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : <Search className="w-5 h-5 stroke-[3]" />}
                            {loading ? "Analyze Record" : "Audit DMARC"}
                        </button>
                    </form>
                </div>

                {/* Error State */}
                <AnimatePresence>
                    {error && (
                        <motion.div
                            initial={{ opacity: 0, y: -10 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -10 }}
                            className="bg-red-50 text-red-700 p-4 rounded-2xl flex items-center gap-3 border border-red-100"
                        >
                            <AlertCircle className="w-5 h-5 flex-shrink-0" />
                            <span className="font-medium">{error}</span>
                        </motion.div>
                    )}
                </AnimatePresence>

                {/* Dashboard & Results */}
                {record && stats && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        className="space-y-8"
                    >
                        {/* 3 Metric Cards */}
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">

                            {/* Policy Score */}
                            <div className="bg-white border border-slate-200 rounded-2xl p-6 shadow-sm flex items-center gap-4">
                                <div className={`w-12 h-12 rounded-xl flex items-center justify-center ${stats.policy === 'reject' ? 'bg-emerald-100 text-emerald-600' :
                                    stats.policy === 'quarantine' ? 'bg-amber-100 text-amber-600' : 'bg-red-100 text-red-600'
                                    }`}>
                                    <Lock className="w-6 h-6" />
                                </div>
                                <div>
                                    <p className="text-slate-500 text-xs font-bold uppercase tracking-wider">Current Policy</p>
                                    <h3 className="text-xl font-black text-slate-900 capitalize">p={stats.policy}</h3>
                                </div>
                            </div>

                            {/* Reporting Status */}
                            <div className="bg-white border border-slate-200 rounded-2xl p-6 shadow-sm flex items-center gap-4">
                                <div className={`w-12 h-12 rounded-xl flex items-center justify-center ${stats.hasRua ? 'bg-blue-100 text-blue-600' : 'bg-slate-100 text-slate-400'
                                    }`}>
                                    <FileText className="w-6 h-6" />
                                </div>
                                <div>
                                    <p className="text-slate-500 text-xs font-bold uppercase tracking-wider">Aggregate Reports</p>
                                    <h3 className="text-xl font-black text-slate-900 capitalize">{stats.hasRua ? "Active" : "Disabled"}</h3>
                                </div>
                            </div>

                            {/* Security Score */}
                            <div className="bg-white border border-slate-200 rounded-2xl p-6 shadow-sm flex items-center gap-4">
                                <div className={`w-12 h-12 rounded-xl flex items-center justify-center ${stats.score >= 80 ? 'bg-emerald-100 text-emerald-600' :
                                    stats.score >= 50 ? 'bg-amber-100 text-amber-600' : 'bg-red-100 text-red-600'
                                    }`}>
                                    <BarChart3 className="w-6 h-6" />
                                </div>
                                <div>
                                    <p className="text-slate-500 text-xs font-bold uppercase tracking-wider">Security Score</p>
                                    <h3 className="text-xl font-black text-slate-900">{stats.score}%</h3>
                                </div>
                            </div>

                        </div>

                        {/* Detailed Breakdown */}
                        <DMARCDeconstructor record={record} />

                        {/* Suggestion / Generator */}
                        <DMARCSuggestion currentPolicy={stats.policy} domain={domain} />

                    </motion.div>
                )}

            </main>


        </div>
    );
}
