"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Sparkles, Copy, Check, ArrowRight } from "lucide-react";

interface DMARCSuggestionProps {
    currentPolicy: string;
    domain: string;
}

export const DMARCSuggestion: React.FC<DMARCSuggestionProps> = ({ currentPolicy, domain }) => {
    const [copied, setCopied] = useState(false);

    // Determine suggestion
    let suggestedPolicy = "reject";
    if (currentPolicy === "none") suggestedPolicy = "quarantine";
    else if (currentPolicy === "quarantine") suggestedPolicy = "reject";
    else return null; // Already at maximum protection

    const suggestedRecord = `v=DMARC1; p=${suggestedPolicy}; rua=mailto:dmarc@${domain}; pct=100; aspf=s; adkim=s;`;

    const copySuggestion = () => {
        navigator.clipboard.writeText(suggestedRecord);
        setCopied(true);
        setTimeout(() => setCopied(false), 2000);
    };

    return (
        <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="bg-gradient-to-br from-blue-600 to-indigo-700 rounded-3xl p-8 text-white relative overflow-hidden shadow-2xl shadow-blue-900/20"
        >
            {/* Background Decor */}
            <div className="absolute top-0 right-0 p-8 opacity-10">
                <Sparkles className="w-32 h-32" />
            </div>

            <div className="relative z-10">
                <div className="flex items-center gap-3 mb-4">
                    <div className="p-2 bg-white/20 backdrop-blur-md rounded-lg">
                        <Sparkles className="w-6 h-6 text-yellow-300" />
                    </div>
                    <h3 className="text-xl font-bold">Optimization Recommendation</h3>
                </div>

                <p className="text-blue-100 mb-6 max-w-2xl font-medium leading-relaxed">
                    Your current policy is set to <span className="font-mono bg-white/10 px-2 py-0.5 rounded font-bold text-white">p={currentPolicy}</span>.
                    We recommend upgrading to <span className="font-bold text-white">p={suggestedPolicy}</span> to
                    {suggestedPolicy === 'quarantine' ? ' start filtering suspicious emails into spam folders.' : ' fully block unauthorized senders.'}
                </p>

                <div className="bg-black/20 backdrop-blur-sm rounded-xl p-4 border border-white/10 flex flex-col md:flex-row md:items-center justify-between gap-4">
                    <code className="font-mono text-sm md:text-base text-blue-100 break-all">
                        {suggestedRecord}
                    </code>

                    <button
                        onClick={copySuggestion}
                        className="flex items-center gap-2 px-5 py-2.5 bg-white text-blue-700 rounded-xl font-bold hover:bg-blue-50 transition-colors shadow-lg whitespace-nowrap"
                    >
                        {copied ? <Check className="w-4 h-4" /> : <Copy className="w-4 h-4" />}
                        Copy Record
                    </button>
                </div>

                <div className="mt-4 flex items-center gap-2 text-xs font-medium text-blue-200 opacity-80">
                    <ArrowRight className="w-3 h-3" />
                    <span>Includes Strict Alignment (aspf=s, adkim=s) and 100% enforcement (pct=100).</span>
                </div>
            </div>
        </motion.div>
    );
};
