"use client";

import React, { useState } from "react";
import { motion, Variants } from "framer-motion";
import { Search, Globe, ChevronDown, CheckCircle, XCircle, Loader2, MapPin } from "lucide-react";

import { checkDnsPropagation, DnsResult } from "@/app/actions/dns";

const recordTypes = ["A", "AAAA", "CNAME", "MX", "NS", "PTR", "TXT"];

export default function DNSLookupClient() {
    const [domain, setDomain] = useState("");
    const [selectedType, setSelectedType] = useState("A");
    const [isSearching, setIsSearching] = useState(false);
    const [results, setResults] = useState<DnsResult[]>([]);

    const handleSearch = async () => {
        if (!domain) return;
        setIsSearching(true);
        // Do not clear results immediately if you want to keep old ones during load, 
        // but clearing makes sense to show new search.
        setResults([]);

        try {
            console.log("Searching for:", domain, selectedType);
            const data = await checkDnsPropagation(domain, selectedType);
            console.log("Received Data:", data);
            setResults(data);
        } catch (error) {
            console.error("DNS Lookup failed:", error);
        } finally {
            setIsSearching(false);
        }
    };

    const containerVariants: Variants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: { staggerChildren: 0.1 }
        }
    };

    const itemVariants: Variants = {
        hidden: { opacity: 0, y: 10 },
        visible: { opacity: 1, y: 0 }
    };

    return (
        <div className="min-h-screen bg-white flex flex-col font-sans text-slate-900">

            {/* Header / Hero */}
            <section className="bg-[#F8FAFC] border-b border-[#E2E8F0] pt-20 pb-16 relative overflow-hidden">
                {/* Map Background Pattern */}
                <div className="absolute inset-0 opacity-10 pointer-events-none" style={{
                    backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%2394a3b8' fill-opacity='0.4'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")`
                }} />

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center relative z-10">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.5 }}
                    >
                        <div className="inline-flex items-center justify-center p-3 bg-blue-50 rounded-2xl mb-6 text-[#2563EB]">
                            <Globe className="w-8 h-8" />
                        </div>
                        <h1 className="text-4xl md:text-5xl font-extrabold text-[#0F172A] mb-6 font-jakarta">
                            Global DNS Propagation Checker
                        </h1>
                        <p className="text-lg text-[#475569] max-w-2xl mx-auto mb-10">
                            Check your DNS records on multiple DNS servers located in different parts of the world.
                        </p>

                        {/* Search Interaction */}
                        <div className="max-w-3xl mx-auto bg-white p-2 rounded-2xl shadow-[0_20px_40px_-15px_rgba(0,0,0,0.05)] border border-[#E2E8F0] flex flex-col sm:flex-row gap-2">

                            {/* Record Type Select */}
                            <div className="relative min-w-[120px]">
                                <select
                                    value={selectedType}
                                    onChange={(e) => setSelectedType(e.target.value)}
                                    className="w-full h-12 pl-4 pr-10 appearance-none bg-slate-50 border border-slate-200 rounded-xl font-bold text-[#0F172A] focus:outline-none focus:ring-2 focus:ring-blue-500/20"
                                >
                                    {recordTypes.map(type => (
                                        <option key={type} value={type}>{type}</option>
                                    ))}
                                </select>
                                <ChevronDown className="absolute right-3 top-1/2 -translate-y-1/2 w-4 h-4 text-slate-500 pointer-events-none" />
                            </div>

                            {/* Input */}
                            <div className="flex-1 relative">
                                <input
                                    type="text"
                                    placeholder="example.com"
                                    value={domain}
                                    onChange={(e) => setDomain(e.target.value)}
                                    onKeyDown={(e) => e.key === "Enter" && handleSearch()}
                                    className="w-full h-12 pl-4 pr-4 bg-transparent text-lg font-medium text-[#0F172A] placeholder-slate-400 outline-none"
                                />
                            </div>

                            {/* Button */}
                            <button
                                onClick={handleSearch}
                                disabled={isSearching}
                                className="h-12 px-8 bg-[#2563EB] hover:bg-[#1E3A8A] text-white font-bold rounded-xl transition-all shadow-lg active:scale-95 disabled:opacity-70 disabled:cursor-not-allowed flex items-center justify-center gap-2"
                            >
                                {isSearching ? <Loader2 className="w-5 h-5 animate-spin" /> : <Search className="w-5 h-5" />}
                                <span className="hidden sm:inline">Search</span>
                            </button>
                        </div>

                    </motion.div>
                </div>
            </section>

            {/* Results Section */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12 w-full relative z-10">

                {results.length === 0 && !isSearching && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        className="flex flex-col items-center justify-center py-20 text-center opacity-50"
                    >
                        <div className="bg-slate-50 p-6 rounded-full mb-4">
                            <MapPin className="w-12 h-12 text-slate-300" />
                        </div>
                        <h3 className="text-xl font-bold text-slate-400">Enter a domain to check propagation</h3>
                        <p className="text-slate-400 mt-2">Check against standard Global DNS Providers</p>
                    </motion.div>
                )}

                {isSearching && results.length === 0 && (
                    <div className="flex flex-col items-center justify-center py-20 text-center">
                        <Loader2 className="w-10 h-10 text-blue-500 animate-spin mb-4" />
                        <p className="text-slate-500 font-medium">Querying Global DNS Servers...</p>
                    </div>
                )}

                {results.length > 0 && (
                    <motion.div
                        className="flex flex-col gap-3"
                        variants={containerVariants}
                        initial="hidden"
                        animate="visible"
                    >
                        {results.map((result, index) => (
                            <motion.div
                                key={`${result.provider}-${index}`}
                                variants={itemVariants}
                                className="bg-white border border-[#E2E8F0] p-4 rounded-xl flex items-center justify-between hover:border-blue-200 hover:shadow-md transition-all group"
                            >
                                <div className="flex items-center gap-4">
                                    <div className="text-3xl filter drop-shadow-sm">{result.flag}</div>
                                    <div>
                                        <div className="font-bold text-[#0F172A] text-sm flex items-center gap-2 group-hover:text-blue-600 transition-colors">
                                            {result.provider}
                                        </div>
                                        <div className="text-slate-500 text-xs font-mono mt-1 flex items-center gap-1">
                                            <Globe className="w-3 h-3" />
                                            {result.location}
                                        </div>
                                    </div>
                                </div>

                                <div className="flex flex-col items-end gap-1 max-w-[60%] sm:max-w-[70%] text-right">
                                    {result.status === "ok" ? (
                                        <>
                                            <div className="flex items-center gap-2 justify-end flex-wrap">
                                                <div className="font-mono text-xs sm:text-sm font-bold text-[#0F172A] bg-slate-100 px-2 py-1 rounded-md text-right break-all">
                                                    {result.value.split(',').map((val, i) => (
                                                        <div key={i}>{val.trim()}</div>
                                                    ))}
                                                </div>
                                                <CheckCircle className="w-5 h-5 text-green-500 shrink-0" />
                                            </div>
                                            <div className="flex items-center gap-3 text-[10px] text-slate-400 font-medium justify-end">
                                                <span>{result.time}</span>
                                                <span>TTL: {result.ttl}</span>
                                            </div>
                                        </>
                                    ) : (
                                        <div className="flex items-center gap-2">
                                            <span className="text-xs text-red-500 font-bold bg-red-50 px-2 py-1 rounded-md">
                                                {result.value}
                                            </span>
                                            <XCircle className="w-5 h-5 text-red-500" />
                                        </div>
                                    )}
                                </div>
                            </motion.div>
                        ))}
                    </motion.div>
                )}

            </main>


        </div>
    );
}
