"use client";

import React from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Check, X, Loader2, Server } from "lucide-react";
import { MapNode } from "./WorldMap";

interface PropagationGridProps {
    nodes: MapNode[];
}

export const PropagationGrid: React.FC<PropagationGridProps> = ({ nodes }) => {

    // Calculate stats
    const resolved = nodes.filter(n => n.status === 'resolved').length;
    const total = nodes.length;
    const percent = Math.round((resolved / total) * 100) || 0;

    return (
        <div className="flex flex-col gap-6">

            {/* Summary Card */}
            <div className="bg-white border border-slate-200 rounded-2xl p-6 shadow-sm flex items-center justify-between">
                <div>
                    <h3 className="text-sm font-bold text-slate-500 uppercase tracking-wider mb-1">Global Reach</h3>
                    <div className="text-3xl font-extrabold text-slate-900">{percent}%</div>
                </div>
                <div className="h-12 w-12 rounded-full bg-blue-50 flex items-center justify-center text-blue-600">
                    <Server className="w-6 h-6" />
                </div>
            </div>

            {/* List */}
            <div className="bg-white border border-slate-200 rounded-2xl shadow-sm overflow-hidden">
                <div className="p-4 bg-slate-50 border-b border-slate-200 text-xs font-bold text-slate-500 uppercase tracking-wider">
                    Propagation Results
                </div>
                <div className="divide-y divide-slate-100">
                    <AnimatePresence>
                        {nodes.map((node, i) => (
                            <motion.div
                                key={node.id}
                                initial={{ opacity: 0, x: -10 }}
                                animate={{ opacity: 1, x: 0 }}
                                transition={{ delay: i * 0.1 }}
                                className="p-4 flex items-center justify-between hover:bg-slate-50 transition-colors group"
                            >
                                <div className="flex items-center gap-3">
                                    <div className="w-8 h-8 rounded-full bg-slate-100 flex items-center justify-center text-lg">
                                        {/* Simple flag mapping based on generic name or just a globe */}
                                        {node.name.includes("USA") ? "🇺🇸" :
                                            node.name.includes("China") ? "🇨🇳" :
                                                node.name.includes("Europe") ? "🇪🇺" :
                                                    node.name.includes("Cyprus") ? "🇨🇾" : "🌍"}
                                    </div>
                                    <div>
                                        <div className="font-bold text-slate-900 text-sm">{node.name}</div>
                                        <div className="text-xs text-slate-500">Node ID: {node.id}</div>
                                    </div>
                                </div>

                                <div className="flex items-center gap-2">
                                    {node.status === 'pending' && (
                                        <Loader2 className="w-5 h-5 text-blue-500 animate-spin" />
                                    )}
                                    {node.status === 'resolved' && (
                                        <div className="flex flex-col items-end">
                                            <span className="font-mono text-xs font-bold text-slate-700 bg-green-50 px-2 py-1 rounded">
                                                {node.ip}
                                            </span>
                                            <Check className="w-4 h-4 text-green-500 mt-1" />
                                        </div>
                                    )}
                                    {node.status === 'error' && (
                                        <div className="flex items-center gap-1 text-red-500">
                                            <span className="text-xs font-bold">Failed</span>
                                            <X className="w-4 h-4" />
                                        </div>
                                    )}
                                </div>
                            </motion.div>
                        ))}
                    </AnimatePresence>
                </div>
            </div>
        </div>
    );
};
