"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { ShieldCheck, Loader2, Globe, Lock, ShieldAlert } from "lucide-react";

import { DNSSECChain, TrustNode } from "@/components/dnssec/DNSSECChain";

interface DNSSECResult {
    status: 'secure' | 'insecure' | 'bogus';
    chain: TrustNode[];
    hasErrors: boolean;
    isEnabled: boolean;
}

export default function DNSSECClient() {
    const [domain, setDomain] = useState("");
    const [loading, setLoading] = useState(false);
    const [result, setResult] = useState<DNSSECResult | null>(null);
    const [error, setError] = useState<string | null>(null);

    const handleValidate = async (e?: React.FormEvent) => {
        if (e) e.preventDefault();
        if (!domain) return;

        setLoading(true);
        setError(null);
        setResult(null);

        try {
            const response = await fetch('/api/dns/dnssec', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ url: domain.trim() }),
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || "Validation failed");
            }

            setResult(data);
        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <ShieldCheck className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                DNSSEC <span className="text-blue-600">Validator</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Verify the cryptographic chain of trust, delegation markers (DS), and digital signatures (RRSIG) for your DNS records.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Search Box */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <form onSubmit={handleValidate} className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1 relative group">
                            <Globe className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                            <input
                                type="text"
                                placeholder="Enter Domain (e.g., icann.org)"
                                value={domain}
                                onChange={(e) => setDomain(e.target.value)}
                                className="w-full h-14 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-2xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                            />
                        </div>

                        <button
                            type="submit"
                            disabled={loading || !domain}
                            className={`h-14 px-8 rounded-2xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95
                                ${loading ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                            `}
                        >
                            {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : <Lock className="w-5 h-5 fill-current" />}
                            {loading ? "Verifying..." : "Verify Trust"}
                        </button>
                    </form>
                </div>

                {loading && (
                    <div className="flex flex-col items-center justify-center py-20 text-center space-y-4">
                        <Loader2 className="w-10 h-10 text-blue-600 animate-spin" />
                        <p className="text-slate-500 font-medium animate-pulse">Tracing Chain of Trust...</p>
                    </div>
                )}

                {result && (
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                    >
                        {/* HERO STATUS CARD */}
                        <div className={`rounded-3xl p-8 mb-8 text-center border-2 
                            ${result.status === 'secure' ? 'bg-emerald-50 border-emerald-500/20' :
                                result.status === 'bogus' ? 'bg-red-50 border-red-500/20' : 'bg-slate-50 border-slate-200'}
                        `}>
                            <div className="inline-flex p-4 rounded-full bg-white shadow-sm mb-4">
                                {result.status === 'secure' && <ShieldCheck className="w-12 h-12 text-emerald-500" />}
                                {result.status === 'bogus' && <ShieldAlert className="w-12 h-12 text-red-500" />}
                                {result.status === 'insecure' && <Lock className="w-12 h-12 text-slate-400" />}
                            </div>
                            <h2 className={`text-3xl font-black mb-2 capitalize
                                ${result.status === 'secure' ? 'text-emerald-900' :
                                    result.status === 'bogus' ? 'text-red-900' : 'text-slate-900'}
                            `}>
                                DNSSEC {result.status}
                            </h2>
                            <p className="text-slate-500 font-medium max-w-lg mx-auto">
                                {result.status === 'secure' ? "This authenticity of this domain is cryptographically verified." :
                                    result.status === 'bogus' ? "CRITICAL: The Chain of Trust is broken. The signatures do not match." :
                                        "This domain does not implement DNSSEC. Records are not signed."}
                            </p>
                        </div>

                        <DNSSECChain chain={result.chain} />
                    </motion.div>
                )}

                {/* Error State */}
                {error && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        className="bg-red-50 text-red-700 p-4 rounded-2xl flex items-center gap-3 border border-red-100"
                    >
                        <ShieldAlert className="w-5 h-5 flex-shrink-0" />
                        <span className="font-medium">{error}</span>
                    </motion.div>
                )}

            </main>


        </div>
    );
}
