"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Activity, Search, Loader2, Globe, AlertCircle, CheckCircle2 } from "lucide-react";

import { EmailHealthDashboard, EmailHealthReport } from "@/components/email-health/EmailHealthDashboard";

export default function EmailHealthClient() {
    const [domain, setDomain] = useState("");
    const [loading, setLoading] = useState(false);
    const [report, setReport] = useState<EmailHealthReport | null>(null);
    const [error, setError] = useState<string | null>(null);

    const handleSearch = async (e?: React.FormEvent) => {
        if (e) e.preventDefault();
        if (!domain) return;

        setLoading(true);
        setError(null);
        setReport(null);

        try {
            const response = await fetch('/api/email-health', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ domain: domain.trim() }),
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || "Audit failed");
            }

            setReport(data);
        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Activity className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                Email Deliverability <span className="text-blue-600">Health</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            A comprehensive check of your mail server configuration, DNS authentication, and sender reputation.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Search Box */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <form onSubmit={handleSearch} className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1 relative group">
                            <Globe className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                            <input
                                type="text"
                                placeholder="Enter Domain (e.g., google.com)"
                                value={domain}
                                onChange={(e) => setDomain(e.target.value)}
                                className="w-full h-14 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-2xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                            />
                        </div>

                        <button
                            type="submit"
                            disabled={loading || !domain}
                            className={`h-14 px-8 rounded-2xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95
                                ${loading ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                            `}
                        >
                            {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : <Activity className="w-5 h-5 stroke-[3]" />}
                            {loading ? "Diagnosing..." : "Run Audit"}
                        </button>
                    </form>
                </div>

                {/* Error State */}
                <AnimatePresence>
                    {error && (
                        <motion.div
                            initial={{ opacity: 0, y: -10 }}
                            animate={{ opacity: 1, y: 0 }}
                            exit={{ opacity: 0, y: -10 }}
                            className="bg-red-50 text-red-700 p-4 rounded-2xl flex items-center gap-3 border border-red-100"
                        >
                            <AlertCircle className="w-5 h-5 flex-shrink-0" />
                            <span className="font-medium">{error}</span>
                        </motion.div>
                    )}
                </AnimatePresence>

                {/* Score Hero & Dashboard */}
                {report && (
                    <div className="space-y-8">
                        {/* Score Hero */}
                        <motion.div
                            initial={{ scale: 0.9, opacity: 0 }}
                            animate={{ scale: 1, opacity: 1 }}
                            className="bg-slate-900 rounded-3xl p-10 text-white shadow-2xl shadow-blue-900/20 flex flex-col items-center justify-center relative overflow-hidden"
                        >
                            <div className="absolute top-0 right-0 p-10 opacity-5">
                                <Activity className="w-64 h-64" />
                            </div>

                            <div className="relative z-10 text-center">
                                <p className="text-slate-400 font-bold uppercase tracking-widest text-sm mb-4">Overall Health Score</p>
                                <div className="text-7xl md:text-9xl font-black mb-2 tracking-tighter bg-clip-text text-transparent bg-gradient-to-br from-white to-slate-400">
                                    {report.score}
                                </div>
                                <div className={`inline-flex items-center gap-2 px-4 py-1.5 rounded-full text-sm font-bold uppercase tracking-wider
                                    ${report.score >= 90 ? 'bg-emerald-500/20 text-emerald-400 border border-emerald-500/30' :
                                        report.score >= 70 ? 'bg-amber-500/20 text-amber-400 border border-amber-500/30' : 'bg-red-500/20 text-red-400 border border-red-500/30'}
                                `}>
                                    {report.score >= 90 ? <CheckCircle2 className="w-4 h-4" /> : <AlertCircle className="w-4 h-4" />}
                                    {report.score >= 90 ? 'Excellent Health' : report.score >= 70 ? 'Action Needed' : 'Critical Issues'}
                                </div>
                            </div>
                        </motion.div>

                        {/* Detailed Dashboard */}
                        <EmailHealthDashboard report={report} />
                    </div>
                )}

            </main>


        </div>
    );
}
