"use client";

import React, { useRef, useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { motion, useScroll, useTransform, useSpring, useMotionValue, useMotionTemplate } from "framer-motion";
import { Search, Globe, Shield, Activity, Server, FileText, Zap, ArrowRight, Command } from "lucide-react";

export const Hero = () => {
    const [query, setQuery] = useState("");
    const router = useRouter();
    const containerRef = useRef<HTMLDivElement>(null);

    // Parallax logic
    const { scrollY } = useScroll();
    const y1 = useTransform(scrollY, [0, 500], [0, 200]);
    const y2 = useTransform(scrollY, [0, 500], [0, -150]);

    const handleSearch = () => {
        if (!query.trim()) return;
        router.push(`/analyze?q=${encodeURIComponent(query)}`);
    };

    return (
        <section
            ref={containerRef}
            className="relative w-full min-h-[95vh] flex items-center justify-center overflow-hidden bg-slate-50 pt-32 lg:pt-0"
        >
            {/* Dynamic Background Mesh */}
            <div className="absolute inset-0 w-full h-full bg-[#FAFAFA]">
                <div className="absolute top-[-10%] left-[-10%] w-[40%] h-[40%] bg-blue-400/20 rounded-full blur-[120px] animate-pulse" />
                <div className="absolute bottom-[-10%] right-[-10%] w-[40%] h-[40%] bg-purple-400/20 rounded-full blur-[120px] animate-pulse delay-1000" />
                <div className="absolute top-[40%] left-[40%] w-[20%] h-[20%] bg-indigo-400/10 rounded-full blur-[80px] animate-pulse delay-700" />
                {/* Grid Pattern */}
                <div className="absolute inset-0 bg-[url('/grid.svg')] opacity-[0.03]" />
            </div>

            <div className="relative z-10 w-full max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 grid grid-cols-1 lg:grid-cols-2 gap-16 lg:gap-0 items-center">

                {/* Left: Content */}
                <motion.div
                    initial={{ opacity: 0, x: -50 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ duration: 0.8, ease: [0.16, 1, 0.3, 1] }}
                    className="space-y-10"
                >
                    {/* Badge */}
                    <div className="inline-flex items-center gap-2 px-4 py-2 rounded-full bg-white/80 backdrop-blur-md border border-slate-200 shadow-sm cursor-default hover:border-blue-200 transition-colors">
                        <span className="relative flex h-2 w-2">
                            <span className="animate-ping absolute inline-flex h-full w-full rounded-full bg-blue-400 opacity-75"></span>
                            <span className="relative inline-flex rounded-full h-2 w-2 bg-blue-500"></span>
                        </span>
                        <span className="text-slate-600 text-sm font-semibold tracking-wide">
                            <span className="text-blue-600">v2.0</span> Now Live
                        </span>
                    </div>

                    {/* Headline */}
                    <div className="space-y-4">
                        <h1 className="text-6xl sm:text-7xl lg:text-8xl font-black text-slate-900 leading-[0.9] tracking-tighter font-jakarta">
                            Master <br />
                            <span className="relative inline-block text-transparent bg-clip-text bg-gradient-to-r from-blue-600 via-indigo-600 to-purple-600 pb-2">
                                Your Domain.
                                <svg className="absolute w-full h-3 -bottom-0 left-0 text-blue-200 -z-10" viewBox="0 0 100 10" preserveAspectRatio="none">
                                    <path d="M0 5 Q 50 10 100 5" stroke="currentColor" strokeWidth="8" fill="none" />
                                </svg>
                            </span>
                        </h1>
                        <p className="text-lg sm:text-xl text-slate-500 font-medium max-w-lg leading-relaxed">
                            The ultimate toolkit for developers. Instant <span className="text-slate-900 font-bold">DNS</span>, <span className="text-slate-900 font-bold">WHOIS</span>, and <span className="text-slate-900 font-bold">Security</span> analysis in one powerful interface.
                        </p>
                    </div>

                    {/* Search Field */}
                    <div className="w-full max-w-xl">
                        <SpotlightSearch
                            query={query}
                            setQuery={setQuery}
                            onSearch={handleSearch}
                        />
                        <div className="mt-4 flex flex-wrap gap-3 text-sm text-slate-400 font-medium">
                            <span>Try:</span>
                            {['google.com', '1.1.1.1', 'github.com'].map(qs => (
                                <button
                                    key={qs}
                                    onClick={() => { setQuery(qs); handleSearch(); }}
                                    className="hover:text-blue-600 hover:underline underline-offset-4 transition-colors"
                                >
                                    {qs}
                                </button>
                            ))}
                        </div>
                    </div>

                    {/* Features List */}
                    <div className="flex gap-8 pt-4 border-t border-slate-200/60">
                        <FeatureItem icon={Zap} text="Real-time Analysis" />
                        <FeatureItem icon={Shield} text="Enterprise Security" />
                        <FeatureItem icon={Globe} text="Global Propagation" />
                    </div>

                </motion.div>

                {/* Right: Visuals */}
                <motion.div
                    style={{ y: y1 }}
                    className="relative hidden lg:flex items-center justify-center pointer-events-none select-none"
                >
                    <div className="relative w-[600px] h-[600px]">
                        {/* Center Orb */}
                        <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[300px] h-[300px] bg-gradient-to-tr from-blue-500 to-purple-500 rounded-full blur-[80px] opacity-20 animate-pulse" />

                        {/* Floating Cards with Tilt */}
                        <TiltCard
                            className="absolute top-[20%] left-[10%] z-20"
                            delay={0.2}
                            icon={Globe}
                            title="DNS Lookup"
                            metric="12ms"
                            status="Active"
                            color="bg-blue-500"
                        />
                        <TiltCard
                            className="absolute top-[10%] right-[15%] z-10"
                            delay={0.4}
                            icon={Shield}
                            title="SSL Secure"
                            metric="256-bit"
                            status="Valid"
                            color="bg-green-500"
                        />
                        <TiltCard
                            className="absolute bottom-[25%] left-[20%] z-30"
                            delay={0.6}
                            icon={Activity}
                            title="Propagation"
                            metric="Global"
                            status="Sync"
                            color="bg-purple-500"
                        />
                        <TiltCard
                            className="absolute bottom-[35%] right-[5%] z-20"
                            delay={0.8}
                            icon={FileText}
                            title="WHOIS Data"
                            metric="Privacy"
                            status="Enabled"
                            color="bg-orange-500"
                        />
                    </div>
                </motion.div>
            </div>
        </section>
    );
};

// --- Subcomponents ---

const FeatureItem = ({ icon: Icon, text }: { icon: any, text: string }) => (
    <div className="flex items-center gap-2 text-slate-500 text-sm font-semibold">
        <div className="bg-slate-100 p-1.5 rounded-lg text-slate-700">
            <Icon className="w-4 h-4" />
        </div>
        {text}
    </div>
);

const SpotlightSearch = ({ query, setQuery, onSearch }: any) => {
    const divRef = useRef<HTMLDivElement>(null);
    const [isFocused, setIsFocused] = useState(false);
    const [position, setPosition] = useState({ x: 0, y: 0 });
    const [opacity, setOpacity] = useState(0);

    const handleMouseMove = (e: React.MouseEvent<HTMLDivElement>) => {
        if (!divRef.current) return;
        const rect = divRef.current.getBoundingClientRect();
        setPosition({ x: e.clientX - rect.left, y: e.clientY - rect.top });
    };

    const handleFocus = () => {
        setIsFocused(true);
        setOpacity(1);
    };

    const handleBlur = () => {
        setIsFocused(false);
        setOpacity(0);
    };

    return (
        <div
            ref={divRef}
            onMouseMove={handleMouseMove}
            onMouseEnter={() => setOpacity(1)}
            onMouseLeave={() => setOpacity(0)}
            className="relative group rounded-2xl p-[2px] overflow-hidden"
        >
            {/* Spotlight Gradient */}
            <div
                className="pointer-events-none absolute -inset-px opacity-0 transition duration-300"
                style={{
                    opacity,
                    background: `radial-gradient(600px circle at ${position.x}px ${position.y}px, rgba(37,99,235,0.15), transparent 40%)`
                }}
            />
            {/* Active Border Gradient */}
            <div
                className={`absolute inset-0 bg-gradient-to-r from-blue-500 via-purple-500 to-blue-500 rounded-2xl opacity-0 transition duration-500 ${isFocused ? 'opacity-100 animate-border-spin' : ''}`}
            />

            <div className="relative bg-white rounded-2xl flex items-center shadow-[0_4px_20px_-2px_rgba(0,0,0,0.1)] border border-slate-200">
                <div className="pl-5 text-slate-400">
                    <Search className={`w-6 h-6 transition-colors ${isFocused ? 'text-blue-600' : ''}`} />
                </div>
                <input
                    type="text"
                    placeholder="Enter domain (e.g., google.com)"
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                    onKeyDown={(e) => e.key === "Enter" && onSearch()}
                    onFocus={handleFocus}
                    onBlur={handleBlur}
                    suppressHydrationWarning
                    className="w-full h-16 bg-transparent px-4 text-xl font-medium text-slate-900 placeholder:text-slate-400 outline-none"
                />
                <button
                    onClick={onSearch}
                    className="mr-2 px-6 py-3 bg-slate-900 hover:bg-slate-800 text-white font-bold rounded-xl transition-all active:scale-95 flex items-center gap-2"
                >
                    Scan <ArrowRight className="w-4 h-4" />
                </button>
            </div>
        </div>
    );
};

const TiltCard = ({ className, delay, icon: Icon, title, metric, status, color }: any) => {
    const x = useMotionValue(0);
    const y = useMotionValue(0);

    const rotateX = useTransform(y, [-100, 100], [30, -30]);
    const rotateY = useTransform(x, [-100, 100], [-30, 30]);

    return (
        <motion.div
            initial={{ opacity: 0, scale: 0.8, y: 50 }}
            animate={{ opacity: 1, scale: 1, y: 0 }}
            transition={{ duration: 0.8, delay, type: "spring" }}
            style={{ x, y, rotateX, rotateY, z: 100 }}
            drag
            dragConstraints={{ top: 0, left: 0, right: 0, bottom: 0 }}
            dragElastic={0.1}
            whileHover={{ scale: 1.05 }}
            className={`cursor-grab active:cursor-grabbing backdrop-blur-xl bg-white/70 border border-white/50 shadow-[0_8px_32px_0_rgba(31,38,135,0.07)] p-4 rounded-3xl w-48 flex items-center gap-4 select-none ${className}`}
        >
            <div className={`p-3 rounded-2xl ${color} bg-opacity-10 text-white shadow-sm`} style={{ backgroundColor: color }}>
                <Icon className="w-5 h-5" />
            </div>
            <div>
                <div className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-0.5">{title}</div>
                <div className="text-slate-900 font-bold text-lg leading-none">{metric}</div>
                <div className="text-[10px] font-bold text-green-600 bg-green-50 px-1.5 py-0.5 rounded mt-1 inline-block">
                    ● {status}
                </div>
            </div>
        </motion.div>
    );
};
