"use client";

import React, { useEffect, useRef } from "react";
import createGlobe from "cobe";
import { useSpring } from "@react-spring/web";

export const LiveGlobe = () => {
    const canvasRef = useRef<HTMLCanvasElement>(null);
    const pointerInteracting = useRef<number | null>(null);
    const pointerInteractionMovement = useRef(0);
    const [{ r }, api] = useSpring(() => ({
        r: 0,
        config: {
            mass: 1,
            tension: 280,
            friction: 40,
            precision: 0.001,
        },
    }));

    useEffect(() => {
        let phi = 0;
        let width = 0;
        const onResize = () => canvasRef.current && (width = canvasRef.current.offsetWidth);
        window.addEventListener('resize', onResize);
        onResize();

        if (!canvasRef.current) return;

        const globe = createGlobe(canvasRef.current, {
            devicePixelRatio: 2,
            width: width * 2,
            height: width * 2,
            phi: 0,
            theta: 0.3,
            dark: 1, // Increased contrast
            diffuse: 3, // Brighter lighting
            mapSamples: 12000,
            mapBrightness: 12,
            baseColor: [1, 1, 1],
            markerColor: [0.1, 0.5, 1], // vivid blue
            glowColor: [0.3, 0.3, 0.3], // subtle glow
            markers: [
                // Major Hubs with pulsing effect simulation (stacked markers)
                { location: [37.7595, -122.4367], size: 0.08 }, // SF
                { location: [37.7595, -122.4367], size: 0.04 },
                { location: [40.7128, -74.006], size: 0.1 },    // NYC
                { location: [40.7128, -74.006], size: 0.05 },
                { location: [51.5074, -0.1278], size: 0.09 },   // London
                { location: [51.5074, -0.1278], size: 0.04 },
                { location: [1.3521, 103.8198], size: 0.08 },   // Singapore
                { location: [35.6762, 139.6503], size: 0.09 },  // Tokyo
                { location: [-33.8688, 151.2093], size: 0.08 }, // Sydney
                { location: [52.5200, 13.4050], size: 0.07 },   // Berlin
                { location: [55.7558, 37.6173], size: 0.06 },   // Moscow
                { location: [-23.5505, -46.6333], size: 0.08 }, // Sao Paulo
                { location: [19.0760, 72.8777], size: 0.07 },   // Mumbai
                { location: [25.2048, 55.2708], size: 0.06 },   // Dubai
            ],
            onRender: (state) => {
                // Drag interaction
                if (!pointerInteracting.current) {
                    phi += 0.005;
                }
                state.phi = phi + r.get();
                state.width = width * 2;
                state.height = width * 2;
            },
        });

        setTimeout(() => (canvasRef.current!.style.opacity = '1'));
        return () => globe.destroy();
    }, []);

    return (
        <div className="relative w-full max-w-[600px] aspect-square mx-auto cursor-grab active:cursor-grabbing">
            <canvas
                ref={canvasRef}
                onPointerDown={(e) => {
                    pointerInteracting.current = e.clientX - pointerInteractionMovement.current;
                    canvasRef.current!.style.cursor = 'grabbing';
                }}
                onPointerUp={() => {
                    pointerInteracting.current = null;
                    canvasRef.current!.style.cursor = 'grab';
                }}
                onPointerOut={() => {
                    pointerInteracting.current = null;
                    canvasRef.current!.style.cursor = 'grab';
                }}
                onMouseMove={(e) => {
                    if (pointerInteracting.current !== null) {
                        const delta = e.clientX - pointerInteracting.current;
                        pointerInteractionMovement.current = delta;
                        api.start({
                            r: delta / 200,
                        });
                    }
                }}
                onTouchMove={(e) => {
                    if (pointerInteracting.current !== null && e.touches[0]) {
                        const delta = e.touches[0].clientX - pointerInteracting.current;
                        pointerInteractionMovement.current = delta;
                        api.start({
                            r: delta / 100,
                        });
                    }
                }}
                style={{
                    width: "100%",
                    height: "100%",
                    contain: "layout paint size",
                    opacity: 0,
                    transition: "opacity 1s ease",
                }}
            />
        </div>
    );
};
