"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Server, Loader2, Search, ArrowRight, ShieldCheck, AlertCircle } from "lucide-react";

import { HeaderDashboard, HeaderResult } from "@/components/http-headers/HeaderDashboard";

export default function HttpHeadersClient() {
    const [url, setUrl] = useState("");
    const [method, setMethod] = useState("GET");
    const [loading, setLoading] = useState(false);
    const [result, setResult] = useState<{
        status: number;
        statusText: string;
        server: string;
        headers: HeaderResult[];
        rawHeaders: string;
    } | null>(null);
    const [error, setError] = useState("");

    const handleAnalyze = async () => {
        if (!url) return;
        setLoading(true);
        setError("");
        setResult(null);

        try {
            const res = await fetch('/api/headers', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ url, method })
            });
            const json = await res.json();

            if (!res.ok) {
                throw new Error(json.error || "Failed to analyze URL");
            }
            setResult(json);

        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Server className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                HTTP Header <span className="text-blue-600">Analyzer</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Inspect raw response headers, security policies, and server details.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Input Card */}
                <div className="bg-white rounded-2xl p-2 shadow-sm border border-slate-200 flex flex-col sm:flex-row gap-2">
                    <select
                        value={method}
                        onChange={(e) => setMethod(e.target.value)}
                        className="h-12 px-4 bg-slate-50 border border-transparent hover:border-slate-200 rounded-xl font-bold text-slate-700 outline-none focus:ring-2 focus:ring-blue-500/20 transition-all cursor-pointer"
                    >
                        <option value="GET">GET</option>
                        <option value="HEAD">HEAD</option>
                        <option value="POST">POST</option>
                    </select>

                    <div className="flex-1 relative group">
                        <input
                            type="text"
                            placeholder="Data (e.g. google.com)"
                            value={url}
                            onChange={(e) => setUrl(e.target.value)}
                            onKeyDown={(e) => e.key === 'Enter' && handleAnalyze()}
                            className="w-full h-12 pl-4 pr-4 bg-slate-50 border border-transparent hover:bg-white hover:border-slate-200 rounded-xl font-medium text-slate-900 placeholder:text-slate-400 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all"
                            suppressHydrationWarning
                        />
                    </div>

                    <button
                        onClick={handleAnalyze}
                        disabled={loading || !url}
                        className="h-12 px-8 bg-slate-900 hover:bg-slate-800 text-white font-bold rounded-xl transition-all shadow-lg active:scale-95 disabled:opacity-70 flex items-center justify-center gap-2"
                    >
                        {loading ? <Loader2 className="w-4 h-4 animate-spin" /> : <>Analyze <ArrowRight className="w-4 h-4" /></>}
                    </button>
                </div>

                {error && (
                    <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} className="bg-red-50 border border-red-100 text-red-600 p-4 rounded-xl text-center font-medium">
                        {error}
                    </motion.div>
                )}

                <AnimatePresence mode="wait">
                    {result && !loading && (
                        <motion.div
                            key="result"
                            initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} exit={{ opacity: 0, scale: 0.95 }}
                            transition={{ duration: 0.4 }}
                            className="space-y-8"
                        >
                            {/* Summary Banner */}
                            <div className={`rounded-xl p-4 border flex items-center justify-between
                                ${result.status >= 200 && result.status < 300 ? "bg-emerald-50 border-emerald-100 text-emerald-800" :
                                    result.status >= 300 && result.status < 400 ? "bg-blue-50 border-blue-100 text-blue-800" :
                                        "bg-red-50 border-red-100 text-red-800"}
                            `}>
                                <div className="flex items-center gap-3">
                                    <div className={`p-2 rounded-lg ${result.status >= 400 ? "bg-red-100" : "bg-white/50"}`}>
                                        {result.status >= 400 ? <AlertCircle className="w-5 h-5" /> : <ShieldCheck className="w-5 h-5" />}
                                    </div>
                                    <div>
                                        <div className="text-xs font-bold uppercase tracking-wider opacity-70">HTTP Status</div>
                                        <div className="text-xl font-black">{result.status} {result.statusText}</div>
                                    </div>
                                </div>
                                <div className="text-right">
                                    <div className="text-xs font-bold uppercase tracking-wider opacity-70">Server Software</div>
                                    <div className="font-mono font-bold">{result.server}</div>
                                </div>
                            </div>

                            <HeaderDashboard headers={result.headers} rawHeaders={result.rawHeaders} />

                        </motion.div>
                    )}
                </AnimatePresence>
            </main>


        </div>
    );
}
