"use client";

import React from "react";
import { motion } from "framer-motion";
import { MapPin, Globe, Shield, Wifi, Server, Activity, Clock, CreditCard, Lock, Eye } from "lucide-react";

export interface IPInfoData {
    ip: string;
    city: string;
    region: string;
    country: string;
    countryCode: string;
    lat: number;
    lon: number;
    asn: string;
    asnName: string;
    asnType?: string; // e.g. isp, business, hosting
    isp: string;
    timezone: string;
    proxy: boolean;
    vpn: boolean;
    tor: boolean;
    hosting: boolean;
    fraudScore: number;
}

interface IPIntelligenceGridProps {
    data: IPInfoData;
}

export const IPIntelligenceGrid: React.FC<IPIntelligenceGridProps> = ({ data }) => {

    // Risk Level Calculation
    const getRiskLevel = (score: number) => {
        if (score < 20) return { label: "SAFE", color: "text-emerald-600", bg: "bg-emerald-50", border: "border-emerald-200", fill: "bg-emerald-500" };
        if (score < 70) return { label: "MODERATE", color: "text-amber-600", bg: "bg-amber-50", border: "border-amber-200", fill: "bg-amber-500" };
        return { label: "HIGH RISK", color: "text-rose-600", bg: "bg-rose-50", border: "border-rose-200", fill: "bg-rose-500" };
    };

    const risk = getRiskLevel(data.fraudScore);
    const localTime = new Date().toLocaleTimeString('en-US', { timeZone: data.timezone, hour: '2-digit', minute: '2-digit' });

    const cardClass = "bg-white border border-slate-200 rounded-3xl p-6 shadow-sm hover:shadow-md hover:border-blue-200 transition-all duration-300 relative overflow-hidden group";

    return (
        <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-4 gap-6 w-full">

            {/* 1. Location Card (Large) */}
            <motion.div
                className={`${cardClass} col-span-1 md:col-span-2 row-span-2`}
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.1 }}
            >
                <div className="flex items-center gap-2 mb-8 text-xs font-bold text-slate-400 uppercase tracking-wider">
                    <MapPin className="w-4 h-4 text-blue-500" /> Geolocation Info
                </div>

                <div className="flex flex-col gap-2 mb-8 relative z-10">
                    <h2 className="text-4xl font-extrabold text-slate-900 leading-tight">{data.country}</h2>
                    <div className="text-xl text-slate-500 font-medium flex items-center gap-2">
                        {data.city}, {data.region}
                        {/* Flag fallback */}
                        <div className="w-8 h-6 rounded overflow-hidden shadow-sm border border-slate-100 ml-2">
                            <img
                                src={`https://flagcdn.com/w80/${data.countryCode.toLowerCase()}.png`}
                                alt={data.country}
                                className="w-full h-full object-cover"
                            />
                        </div>
                    </div>
                </div>

                <div className="grid grid-cols-2 gap-8 relative z-10">
                    <div>
                        <div className="text-xs font-bold text-slate-400 mb-1 uppercase">Coordinates</div>
                        <div className="font-mono text-slate-700 font-bold bg-slate-50 inline-block px-2 py-1 rounded-lg border border-slate-100">
                            {data.lat.toFixed(4)}, {data.lon.toFixed(4)}
                        </div>
                    </div>
                    <div>
                        <div className="text-xs font-bold text-slate-400 mb-1 uppercase">Local Time</div>
                        <div className="font-mono text-slate-700 font-bold flex items-center gap-2">
                            <Clock className="w-4 h-4 text-blue-500" />
                            {localTime}
                        </div>
                    </div>
                </div>

                {/* Map Placeholder Graphic */}
                <div className="absolute -bottom-10 -right-10 opacity-5 group-hover:opacity-10 transition-opacity duration-500 pointer-events-none">
                    <Globe className="w-64 h-64 text-slate-900" />
                </div>
            </motion.div>

            {/* 2. Security & Fraud Score */}
            <motion.div
                className={`${cardClass} col-span-1 md:col-span-1 lg:col-span-2 ${risk.bg} ${risk.border} border-2`}
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.2 }}
            >
                <div className="flex items-center justify-between mb-6">
                    <div className="flex items-center gap-2 text-xs font-bold text-slate-500 uppercase tracking-wider">
                        <Shield className="w-4 h-4" /> Security Analysis
                    </div>
                    <div className={`px-3 py-1 rounded-full text-[10px] font-black tracking-widest ${risk.bg} ${risk.color} border ${risk.border}`}>
                        {risk.label}
                    </div>
                </div>

                <div className="flex items-end gap-2 mb-2">
                    <span className="text-6xl font-black text-slate-900 leading-none">{data.fraudScore}</span>
                    <span className="text-sm font-bold text-slate-400 mb-2">/ 100 Risk Score</span>
                </div>

                {/* Visual Bar */}
                <div className="w-full h-2 bg-white/50 rounded-full overflow-hidden mb-6">
                    <motion.div
                        initial={{ width: 0 }} animate={{ width: `${data.fraudScore}%` }} transition={{ duration: 1, ease: "circOut" }}
                        className={`h-full rounded-full ${risk.fill}`}
                    />
                </div>

                <div className="flex flex-wrap gap-2">
                    <Badge label="PROXY" active={data.proxy} icon={<Eye className="w-3 h-3" />} />
                    <Badge label="VPN" active={data.vpn} icon={<Lock className="w-3 h-3" />} />
                    <Badge label="TOR" active={data.tor} icon={<Activity className="w-3 h-3" />} />
                    <Badge label="HOSTING" active={data.hosting} icon={<Server className="w-3 h-3" />} />
                </div>
            </motion.div>

            {/* 3. Network / ASN */}
            <motion.div
                className={`${cardClass} col-span-1 md:col-span-3 lg:col-span-2`}
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.3 }}
            >
                <div className="flex items-center gap-2 mb-6 text-xs font-bold text-slate-400 uppercase tracking-wider">
                    <Server className="w-4 h-4 text-purple-500" /> Network Identity
                </div>

                <div className="flex flex-col gap-6">
                    <div>
                        <div className="text-xs font-bold text-slate-400 mb-1 uppercase">Internet Service Provider</div>
                        <div className="text-xl font-bold text-slate-900">{data.isp}</div>
                    </div>

                    <div className="grid grid-cols-2 gap-4">
                        <div className="bg-slate-50 p-3 rounded-xl border border-slate-100">
                            <div className="text-[10px] font-bold text-slate-400 mb-1 uppercase">ASN</div>
                            <div className="font-mono text-blue-600 font-bold">{data.asn}</div>
                        </div>
                        <div className="bg-slate-50 p-3 rounded-xl border border-slate-100">
                            <div className="text-[10px] font-bold text-slate-400 mb-1 uppercase">Organization</div>
                            <div className="font-medium text-slate-700 text-sm line-clamp-1">{data.asnName}</div>
                        </div>
                    </div>
                </div>
            </motion.div>

            {/* 4. Technical Details */}
            <motion.div
                className={`${cardClass} col-span-1 md:col-span-3 lg:col-span-2`}
                initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }} transition={{ delay: 0.4 }}
            >
                <div className="flex items-center gap-2 mb-6 text-xs font-bold text-slate-400 uppercase tracking-wider">
                    <Activity className="w-4 h-4 text-orange-500" /> Technical Profile
                </div>

                <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
                    <div>
                        <div className="text-xs font-bold text-slate-400 mb-1 uppercase">Timezone</div>
                        <div className="font-medium text-slate-700 text-sm">{data.timezone}</div>
                    </div>
                    <div>
                        <div className="text-xs font-bold text-slate-400 mb-1 uppercase">Currency</div>
                        <div className="font-medium text-slate-700 text-sm flex items-center gap-1">
                            <CreditCard className="w-3 h-3 text-slate-400" />
                            Unknown
                        </div>
                    </div>
                    <div className="col-span-2 bg-slate-50 rounded-xl flex items-center justify-center p-2 border border-slate-100">
                        <div className="flex items-center gap-2 text-sm font-semibold text-slate-600">
                            <Wifi className="w-4 h-4 text-slate-400" />
                            {data.hosting ? "Commercial / Business" : "Residential / Private"}
                        </div>
                    </div>
                </div>
            </motion.div>

        </div>
    );
};

const Badge = ({ label, active, icon }: { label: string, active: boolean, icon: React.ReactNode }) => (
    <div className={`flex items-center gap-1.5 px-3 py-1.5 rounded-lg text-[10px] font-bold border transition-colors
        ${active ? "bg-red-50 border-red-200 text-red-600" : "bg-white border-slate-200 text-slate-400 opacity-60 grayscale"}
    `}>
        {icon}
        {label}
    </div>
);
