"use client";

import React, { useMemo } from "react";
import { AreaChart, Area, XAxis, YAxis, Tooltip, ResponsiveContainer, CartesianGrid } from "recharts";
import { Activity } from "lucide-react";

interface PingResult {
    seq: number;
    time: number;
    ttl: number;
    status: 'success' | 'timeout' | 'error';
}

interface PingChartProps {
    data: PingResult[];
}

export const PingChart: React.FC<PingChartProps> = ({ data }) => {

    // Calculate Jitter (Standard Deviation of latencies)
    const jitter = useMemo(() => {
        if (data.length < 2) return 0;
        const validPings = data.filter(d => d.status === 'success' && d.time > 0).map(d => d.time);
        if (validPings.length < 2) return 0;

        const mean = validPings.reduce((a, b) => a + b, 0) / validPings.length;
        const variance = validPings.reduce((a, b) => a + Math.pow(b - mean, 2), 0) / validPings.length;
        return Math.sqrt(variance);
    }, [data]);

    // Format data for chart (filtering out timeouts/errors for the line, or treating them as 0/null?)
    // Better to just show the valid ones or gaps.
    const chartData = data.map(d => ({
        seq: d.seq,
        time: d.status === 'success' ? d.time : null
    }));

    // Calculate nice max for Y axis
    const maxTime = Math.max(...chartData.map(d => d.time || 0), 10);

    return (
        <div className="bg-white border border-slate-200 rounded-3xl p-6 shadow-xl shadow-blue-500/5 relative overflow-hidden h-full flex flex-col">

            <div className="flex items-center justify-between mb-6">
                <div className="flex items-center gap-3">
                    <div className="p-2 bg-blue-50 rounded-lg">
                        <Activity className="w-5 h-5 text-blue-600" />
                    </div>
                    <div>
                        <h3 className="text-lg font-bold text-slate-900 leading-none">Latency Visualizer</h3>
                        <p className="text-xs text-slate-500 font-medium mt-1">Real-time Response Time</p>
                    </div>
                </div>

                <div className="flex flex-col items-end">
                    <span className="text-xs font-bold text-slate-400 uppercase tracking-wider">Jitter</span>
                    <span className={`text-xl font-black font-mono ${jitter > 10 ? "text-amber-500" : "text-slate-900"}`}>
                        {jitter.toFixed(1)}ms
                    </span>
                </div>
            </div>

            <div className="w-full h-[300px]">
                <ResponsiveContainer width="100%" height="100%">
                    <AreaChart data={chartData}>
                        <defs>
                            <linearGradient id="colorLatency" x1="0" y1="0" x2="0" y2="1">
                                <stop offset="5%" stopColor="#2563EB" stopOpacity={0.3} />
                                <stop offset="95%" stopColor="#2563EB" stopOpacity={0} />
                            </linearGradient>
                        </defs>
                        <CartesianGrid strokeDasharray="3 3" vertical={false} stroke="#f1f5f9" />
                        <XAxis
                            dataKey="seq"
                            hide={true} // Hide X axis labels for cleaner look, seq is generic
                        />
                        <YAxis
                            unit="ms"
                            stroke="#94a3b8"
                            fontSize={10}
                            tickLine={false}
                            axisLine={false}
                            domain={[0, (dataMax: number) => Math.max(dataMax * 1.2, 50)]} // Add some headroom
                        />
                        <Tooltip
                            contentStyle={{ borderRadius: '12px', border: 'none', boxShadow: '0 10px 15px -3px rgba(0, 0, 0, 0.1)' }}
                            itemStyle={{ color: '#0f172a', fontWeight: 'bold' }}
                            labelStyle={{ display: 'none' }}
                            formatter={(value: any) => [`${Number(value).toFixed(1)} ms`, 'Latency']}
                        />
                        <Area
                            type="monotone"
                            dataKey="time"
                            stroke="#2563EB"
                            fillOpacity={1}
                            fill="url(#colorLatency)"
                            strokeWidth={3}
                            animationDuration={300}
                        />
                    </AreaChart>
                </ResponsiveContainer>
            </div>

        </div>
    );
};
