"use client";

import React from "react";
import { motion } from "framer-motion";
import { ArrowDown, Copy, AlertTriangle, CheckCircle, Clock, Link as LinkIcon, RotateCw } from "lucide-react";

export interface RedirectHop {
    url: string;
    status: number;
    statusText: string;
    timing: number;
    location: string | null;
    server?: string;
}

interface RedirectChainProps {
    hops: RedirectHop[];
    isLoop: boolean;
}

export const RedirectChain: React.FC<RedirectChainProps> = ({ hops, isLoop }) => {

    const copyToClipboard = (text: string) => {
        navigator.clipboard.writeText(text);
    };

    const getStatusColor = (status: number) => {
        if (status >= 200 && status < 300) return "bg-emerald-100 text-emerald-700 border-emerald-200";
        if (status === 301) return "bg-blue-100 text-blue-700 border-blue-200";
        if (status === 302) return "bg-purple-100 text-purple-700 border-purple-200";
        if (status >= 400) return "bg-red-100 text-red-700 border-red-200";
        return "bg-slate-100 text-slate-700 border-slate-200";
    };

    return (
        <div className="space-y-4 max-w-3xl mx-auto py-8">

            {hops.map((hop, index) => {
                const isFirst = index === 0;
                const isLast = index === hops.length - 1;

                return (
                    <motion.div
                        initial={{ opacity: 0, y: 10 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ delay: index * 0.1 }}
                        key={index}
                        className="relative pl-8 md:pl-0"
                    >
                        {/* Timeline Connector */}
                        {!isLast && (
                            <div className="absolute left-6 md:left-1/2 md:-translate-x-1/2 top-16 bottom-[-2rem] w-0.5 border-l-2 border-dashed border-slate-200 z-0 h-full"></div>
                        )}

                        {/* Card */}
                        <div className="bg-white border border-slate-200 rounded-2xl p-6 shadow-sm relative z-10 md:w-full">
                            <div className="flex flex-col md:flex-row gap-4 md:items-center justify-between">

                                <div className="flex items-start gap-4 flex-1 overflow-hidden">
                                    {/* Status Badge */}
                                    <div className={`flex-shrink-0 w-16 h-12 flex items-center justify-center rounded-xl border font-black text-sm ${getStatusColor(hop.status)}`}>
                                        {hop.status}
                                    </div>

                                    <div className="min-w-0 flex-1">
                                        <div className="flex items-center gap-2 mb-1">
                                            <p className="text-xs font-bold text-slate-500 uppercase tracking-wider">
                                                {isFirst ? "Initial Request" : `Hop #${index + 1}`}
                                            </p>
                                            {hop.statusText && <span className="text-xs text-slate-400 font-medium px-1.5 py-0.5 bg-slate-100 rounded">{hop.statusText}</span>}
                                        </div>

                                        <div className="flex items-center gap-2 group">
                                            <p className="font-mono text-sm text-slate-800 truncate" title={hop.url}>
                                                {hop.url.length > 60 ? hop.url.substring(0, 60) + '...' : hop.url}
                                            </p>
                                            <button
                                                onClick={() => copyToClipboard(hop.url)}
                                                className="opacity-0 group-hover:opacity-100 transition-opacity text-slate-400 hover:text-blue-600"
                                            >
                                                <Copy className="w-3.5 h-3.5" />
                                            </button>
                                        </div>

                                        {hop.server && (
                                            <p className="text-xs text-slate-400 mt-1">Server: {hop.server}</p>
                                        )}
                                    </div>
                                </div>

                                {/* Timing */}
                                <div className="flex items-center gap-1.5 text-xs font-bold text-slate-400 bg-slate-50 px-3 py-1.5 rounded-lg border border-slate-100 whitespace-nowrap">
                                    <Clock className="w-3.5 h-3.5" />
                                    {hop.timing}ms
                                </div>

                            </div>

                            {/* Redirect Arrow Highlight */}
                            {hop.location && (
                                <div className="mt-4 pt-3 border-t border-slate-50 flex items-center gap-2 text-xs font-medium text-slate-500">
                                    <ArrowDown className="w-3.5 h-3.5 text-blue-500" />
                                    Redirects to: <span className="font-mono text-blue-600">{new URL(hop.location, hop.url).pathname}</span> ...
                                </div>
                            )}
                        </div>

                        {/* Arrow Icon between cards visual */}
                        {!isLast && (
                            <div className="flex justify-center my-2 text-slate-300 relative z-10 md:hidden">
                                <ArrowDown className="w-5 h-5 animate-bounce" />
                            </div>
                        )}
                        {!isLast && (
                            <div className="hidden md:flex justify-center -mb-5 relative z-20 mt-2">
                                <div className="bg-slate-50 p-1.5 rounded-full border border-slate-200">
                                    <ArrowDown className="w-4 h-4 text-slate-400" />
                                </div>
                            </div>
                        )}

                    </motion.div>
                );
            })}

            {/* Loop Alert */}
            {isLoop && (
                <motion.div
                    initial={{ opacity: 0, scale: 0.9 }}
                    animate={{ opacity: 1, scale: 1 }}
                    className="mt-8 p-6 bg-red-50 border border-red-200 rounded-3xl flex items-center gap-4 text-red-800"
                >
                    <div className="p-3 bg-red-100 rounded-full">
                        <RotateCw className="w-6 h-6 text-red-600 animate-spin-slow" />
                    </div>
                    <div>
                        <h3 className="font-bold text-lg">Infinite Loop Detected</h3>
                        <p className="text-sm text-red-700/80 font-medium">
                            The browser will stop following redirects after ~20 hops. This chain directs back to a previous URL.
                        </p>
                    </div>
                </motion.div>
            )}

            {/* Success State */}
            {!isLoop && hops.length > 0 && hops[hops.length - 1].status < 400 && (
                <motion.div
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    className="mt-8 flex flex-col items-center text-center space-y-2 p-6 rounded-3xl bg-emerald-50/50 border border-emerald-100"
                >
                    <div className="p-3 bg-emerald-100 text-emerald-600 rounded-full mb-2">
                        <CheckCircle className="w-6 h-6" />
                    </div>
                    <h3 className="font-bold text-lg text-emerald-900">Destination Reached</h3>
                    <p className="text-sm text-emerald-700/80">
                        The redirect chain resolved successfully to a valid page.
                    </p>
                </motion.div>
            )}
        </div>
    );
};
