"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Link as LinkIcon, Loader2, ChevronRight, Monitor, Smartphone, AlertCircle } from "lucide-react";

import { RedirectChain, RedirectHop } from "@/components/redirect-checker/RedirectChain";

export default function RedirectClient() {
    const [url, setUrl] = useState("");
    const [isMobile, setIsMobile] = useState(false);
    const [loading, setLoading] = useState(false);
    const [hops, setHops] = useState<RedirectHop[]>([]);
    const [isLoop, setIsLoop] = useState(false);
    const [error, setError] = useState<string | null>(null);

    const handleTrace = async (e?: React.FormEvent) => {
        if (e) e.preventDefault();
        if (!url) return;

        setLoading(true);
        setError(null);
        setHops([]);
        setIsLoop(false);

        try {
            const response = await fetch('/api/trace/redirect', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    url: url.trim(),
                    device: isMobile ? 'mobile' : 'desktop'
                }),
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || "Trace failed");
            }

            setHops(data.hops);
            setIsLoop(data.isLoop);
        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    const totalTime = hops.reduce((sum, h) => sum + h.timing, 0);
    const finalHop = hops.length > 0 ? hops[hops.length - 1] : null;

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <LinkIcon className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                Redirect Chain <span className="text-blue-600">Analyzer</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Trace URL redirection paths, identify status codes, headers, and detect infinite loops or broken links.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Search Box */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <form onSubmit={handleTrace} className="flex flex-col gap-4">
                        <div className="flex flex-col md:flex-row gap-4">
                            <div className="flex-1 relative group">
                                <LinkIcon className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                                <input
                                    type="text"
                                    placeholder="Enter URL to Trace (e.g., bit.ly/example)"
                                    value={url}
                                    onChange={(e) => setUrl(e.target.value)}
                                    className="w-full h-14 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-2xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                                />
                            </div>

                            <button
                                type="submit"
                                disabled={loading || !url}
                                className={`h-14 px-8 rounded-2xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95 whitespace-nowrap
                                    ${loading ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                                `}
                            >
                                {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : <ChevronRight className="w-5 h-5 stroke-[3]" />}
                                {loading ? "Tracing..." : "Trace Path"}
                            </button>
                        </div>

                        {/* Device Toggle */}
                        <div className="flex items-center gap-4">
                            <span className="text-xs font-bold text-slate-400 uppercase tracking-wider">Simulate Device:</span>
                            <button
                                type="button"
                                onClick={() => setIsMobile(false)}
                                className={`flex items-center gap-2 px-3 py-1.5 rounded-lg text-sm font-bold transition-all border
                                    ${!isMobile ? "bg-blue-50 text-blue-700 border-blue-200" : "bg-white text-slate-500 border-slate-200 hover:bg-slate-50"}
                                `}
                            >
                                <Monitor className="w-4 h-4" /> Desktop
                            </button>
                            <button
                                type="button"
                                onClick={() => setIsMobile(true)}
                                className={`flex items-center gap-2 px-3 py-1.5 rounded-lg text-sm font-bold transition-all border
                                    ${isMobile ? "bg-blue-50 text-blue-700 border-blue-200" : "bg-white text-slate-500 border-slate-200 hover:bg-slate-50"}
                                `}
                            >
                                <Smartphone className="w-4 h-4" /> Mobile
                            </button>
                        </div>
                    </form>
                </div>

                {hops.length > 0 && (
                    <motion.div
                        initial={{ opacity: 0, y: 10 }}
                        animate={{ opacity: 1, y: 0 }}
                        className="grid grid-cols-2 md:grid-cols-4 gap-4"
                    >
                        {/* Stat Cards */}
                        <div className="bg-white p-5 rounded-2xl border border-slate-200 shadow-sm">
                            <p className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-1">Total Hops</p>
                            <p className="text-2xl font-black text-slate-900">{hops.length}</p>
                        </div>
                        <div className="bg-white p-5 rounded-2xl border border-slate-200 shadow-sm">
                            <p className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-1">Total Time</p>
                            <p className="text-2xl font-black text-slate-900">{totalTime}ms</p>
                        </div>
                        <div className="bg-white p-5 rounded-2xl border border-slate-200 shadow-sm col-span-2">
                            <p className="text-xs font-bold text-slate-400 uppercase tracking-widest mb-1">Final Status</p>
                            <p className={`text-2xl font-black ${finalHop?.status && finalHop.status >= 400 ? 'text-red-600' : 'text-emerald-600'
                                }`}>
                                {finalHop ? `${finalHop.status} ${finalHop.statusText}` : '-'}
                            </p>
                        </div>
                    </motion.div>
                )}

                {/* Main Visualizer */}
                {hops.length > 0 && (
                    <RedirectChain hops={hops} isLoop={isLoop} />
                )}

                {/* Error State */}
                {error && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        className="bg-red-50 text-red-700 p-4 rounded-2xl flex items-center gap-3 border border-red-100"
                    >
                        <AlertCircle className="w-5 h-5 flex-shrink-0" />
                        <span className="font-medium">{error}</span>
                    </motion.div>
                )}

            </main>


        </div>
    );
}
