"use client";

import React from "react";
import { motion } from "framer-motion";
import { ArrowRight, Server, Globe, Shield, Wifi, Copy, Check } from "lucide-react";
import Link from "next/link";

export interface PTRResult {
    input: string;
    ip: string;
    hostname: string[] | null;
    arpa: string;
    type: 'IPv4' | 'IPv6';
    isDomain?: boolean;
}

interface PTRResultCardProps {
    result: PTRResult;
}

export const PTRResultCard: React.FC<PTRResultCardProps> = ({ result }) => {
    const [copied, setCopied] = React.useState(false);

    const handleCopy = () => {
        if (result.hostname && result.hostname.length > 0) {
            navigator.clipboard.writeText(result.hostname[0]);
            setCopied(true);
            setTimeout(() => setCopied(false), 2000);
        }
    };

    const hasResult = result.hostname && result.hostname.length > 0;

    return (
        <div className="space-y-8">

            {/* Main Directional Card */}
            <div className="bg-white border border-slate-200 rounded-3xl p-8 shadow-xl shadow-blue-500/5 relative overflow-hidden">
                <div className="absolute top-0 inset-x-0 h-1 bg-gradient-to-r from-blue-500 via-indigo-500 to-blue-500" />

                <div className="flex flex-col lg:flex-row items-center justify-between gap-8">

                    {/* 1. Source (Domain or IP) */}
                    <div className="flex flex-col items-center lg:items-start text-center lg:text-left flex-1 min-w-0">
                        <div className="p-3 bg-slate-50 rounded-xl mb-4 border border-slate-100 relative">
                            <Wifi className="w-8 h-8 text-slate-400" />
                            {result.isDomain && (
                                <div className="absolute -top-2 -right-2 bg-blue-100 text-blue-600 text-[10px] font-bold px-2 py-0.5 rounded-full border border-blue-200">
                                    DOMAIN
                                </div>
                            )}
                        </div>
                        <div className="text-sm font-bold text-slate-400 uppercase tracking-wider mb-1">
                            {result.isDomain ? "Resolved Domain" : "Source IP"}
                        </div>
                        <div className="text-2xl font-black text-slate-800 font-mono break-all hover:text-blue-600 transition-colors cursor-default">
                            {result.input}
                        </div>
                        {result.isDomain && (
                            <div className="mt-1 flex items-center gap-2 text-sm text-slate-500 font-mono">
                                <ArrowRight className="w-3 h-3 text-slate-300" /> {result.ip}
                            </div>
                        )}
                        {!result.isDomain && (
                            <div className="inline-flex mt-2 px-2 py-0.5 rounded text-xs font-bold bg-slate-100 text-slate-500 border border-slate-200">
                                {result.type} Address
                            </div>
                        )}
                    </div>

                    {/* 2. Directional Arrow */}
                    <div className="flex flex-col items-center justify-center shrink-0">
                        <div className="w-12 h-12 rounded-full bg-blue-50 flex items-center justify-center text-blue-500 animate-pulse">
                            <ArrowRight className="w-6 h-6" />
                        </div>
                        <div className="text-[10px] font-bold text-blue-400 uppercase tracking-widest mt-2">Resolves To</div>
                    </div>

                    {/* 3. Result Hostname */}
                    <div className="flex flex-col items-center lg:items-end text-center lg:text-right flex-1 min-w-0">
                        <div className={`p-3 rounded-xl mb-4 border ${hasResult ? "bg-emerald-50 border-emerald-100" : "bg-slate-50 border-slate-100"}`}>
                            <Server className={`w-8 h-8 ${hasResult ? "text-emerald-500" : "text-slate-300"}`} />
                        </div>
                        <div className="text-sm font-bold text-slate-400 uppercase tracking-wider mb-1">PTR Record Target</div>

                        {hasResult ? (
                            <div className="group relative flex items-center gap-2">
                                <div className="text-2xl font-black text-slate-900 font-mono break-all text-right">
                                    {result.hostname![0]}
                                </div>
                                <button onClick={handleCopy} className="p-1 text-slate-300 hover:text-blue-500 transition-colors">
                                    {copied ? <Check className="w-4 h-4" /> : <Copy className="w-4 h-4" />}
                                </button>
                            </div>
                        ) : (
                            <div className="px-4 py-2 bg-slate-100 rounded-lg border border-slate-200 text-slate-500 font-medium italic">
                                No PTR Record Found
                            </div>
                        )}

                        {hasResult && result.hostname!.length > 1 && (
                            <div className="mt-2 text-xs text-slate-400">
                                + {result.hostname!.length - 1} additional records
                            </div>
                        )}
                    </div>

                </div>

                {/* Technical ARPA Footer */}
                <div className="mt-8 pt-6 border-t border-slate-100">
                    <div className="flex flex-col sm:flex-row items-center justify-between gap-4 text-xs">
                        <div className="font-mono text-slate-400 break-all">
                            <span className="font-bold text-slate-500">Query:</span> {result.arpa}
                        </div>
                        <div className="flex items-center gap-1 text-slate-400">
                            <Check className="w-3 h-3 text-emerald-500" /> DNS RFC Standard 1035
                        </div>
                    </div>
                </div>
            </div>

            {/* Quick Actions */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <Link href={`/whois?q=${result.ip}`} className="group p-4 bg-white border border-slate-200 rounded-2xl hover:border-blue-200 hover:shadow-lg hover:shadow-blue-500/5 transition-all flex items-center gap-4">
                    <div className="p-3 bg-blue-50 text-blue-600 rounded-xl group-hover:scale-110 transition-transform">
                        <Globe className="w-5 h-5" />
                    </div>
                    <div>
                        <div className="font-bold text-slate-900 group-hover:text-blue-600 transition-colors">Whois Lookup</div>
                        <div className="text-xs text-slate-500">Check ownership details for {result.ip}</div>
                    </div>
                </Link>

                <Link href={`/blacklist-checker?q=${result.ip}`} className="group p-4 bg-white border border-slate-200 rounded-2xl hover:border-red-200 hover:shadow-lg hover:shadow-red-500/5 transition-all flex items-center gap-4">
                    <div className="p-3 bg-red-50 text-red-600 rounded-xl group-hover:scale-110 transition-transform">
                        <Shield className="w-5 h-5" />
                    </div>
                    <div>
                        <div className="font-bold text-slate-900 group-hover:text-red-600 transition-colors">Blacklist Audit</div>
                        <div className="text-xs text-slate-500">Is {result.ip} flagged as spam?</div>
                    </div>
                </Link>
            </div>

        </div>
    );
};
