"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Wifi, Loader2, ArrowRight, Search } from "lucide-react";

import { PTRResult, PTRResultCard } from "@/components/reverse-dns/PTRResultCard";

export default function ReverseDNSClient() {
    const [ip, setIp] = useState("");
    const [loading, setLoading] = useState(false);
    const [result, setResult] = useState<PTRResult | null>(null);
    const [error, setError] = useState("");

    const handleLookup = async () => {
        if (!ip) return;
        setLoading(true);
        setError("");
        setResult(null);

        try {
            const res = await fetch('/api/dns/reverse', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ ip })
            });
            const json = await res.json();

            if (!res.ok) {
                throw new Error(json.error || "Failed to perform reverse lookup");
            }
            setResult(json);

        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Wifi className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                Reverse DNS <span className="text-blue-600">Lookup</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Reveal the hostname associated with any IPv4 or IPv6 address using system-level PTR records.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-12">

                {/* Search Box */}
                <div className="max-w-2xl mx-auto">
                    <div className="bg-white rounded-2xl p-2 shadow-xl shadow-blue-900/5 border border-slate-200 flex flex-col sm:flex-row gap-2">
                        <div className="flex-1 relative group">
                            <Search className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                            <input
                                type="text"
                                placeholder="Enter IP address or Domain (e.g. google.com)"
                                value={ip}
                                onChange={(e) => setIp(e.target.value)}
                                onKeyDown={(e) => e.key === 'Enter' && handleLookup()}
                                className="w-full h-12 pl-12 pr-4 bg-slate-50 border border-transparent hover:bg-white hover:border-slate-200 rounded-xl font-medium text-slate-900 placeholder:text-slate-400 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-mono"
                                suppressHydrationWarning
                            />
                        </div>

                        <button
                            onClick={handleLookup}
                            disabled={loading || !ip}
                            className="h-12 px-8 bg-blue-600 hover:bg-blue-700 text-white font-bold rounded-xl transition-all shadow-lg shadow-blue-600/20 active:scale-95 disabled:opacity-70 disabled:active:scale-100 flex items-center justify-center gap-2"
                        >
                            {loading ? <Loader2 className="w-4 h-4 animate-spin" /> : <>Lookup <ArrowRight className="w-4 h-4" /></>}
                        </button>
                    </div>
                    <div className="text-center mt-3 text-xs text-slate-400 font-medium">
                        Supports IPv4 and IPv6 Address Formats
                    </div>
                </div>

                {error && (
                    <motion.div initial={{ opacity: 0 }} animate={{ opacity: 1 }} className="max-w-2xl mx-auto bg-red-50 border border-red-100 text-red-600 p-4 rounded-xl text-center font-medium">
                        {error}
                    </motion.div>
                )}

                <AnimatePresence mode="wait">
                    {result && !loading && (
                        <motion.div
                            key="result"
                            initial={{ opacity: 0, y: 20 }} animate={{ opacity: 1, y: 0 }} exit={{ opacity: 0, scale: 0.95 }}
                            transition={{ duration: 0.4 }}
                        >
                            <PTRResultCard result={result} />
                        </motion.div>
                    )}
                </AnimatePresence>
            </main>


        </div>
    );
}
