"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Server, Play, Loader2, Shield, Settings, Globe } from "lucide-react";

import { SMTPSessionConsole, SMTPSessionStep } from "@/components/smtp/SMTPSessionConsole";

export default function SMTPClient() {
    const [host, setHost] = useState("");
    const [port, setPort] = useState(587); // Default
    const [encryption, setEncryption] = useState("starttls");

    const [loading, setLoading] = useState(false);
    const [session, setSession] = useState<SMTPSessionStep[]>([]);
    const [tlsEstablished, setTlsEstablished] = useState(false);
    const [banner, setBanner] = useState<string | null>(null);

    const runTest = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!host) return;

        setLoading(true);
        setSession([]);
        setTlsEstablished(false);
        setBanner(null);

        try {
            const res = await fetch('/api/smtp/test', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ host, port, encryption }),
            });
            const data = await res.json();

            if (data.session) setSession(data.session);
            if (data.tlsEstablished) setTlsEstablished(data.tlsEstablished);
            if (data.banner) setBanner(data.banner);

        } catch (err) {
            console.error(err);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Server className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                SMTP Server <span className="text-blue-600">Diagnostics</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Test mail server connectivity, verify handshakes, and debug SMTP response codes in real-time.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full">
                <div className="flex flex-col lg:flex-row gap-8">

                    {/* Left: Input Controls */}
                    <motion.div
                        initial={{ opacity: 0, x: -10 }}
                        animate={{ opacity: 1, x: 0 }}
                        className="lg:w-1/3 flex flex-col gap-6"
                    >
                        <form onSubmit={runTest} className="bg-white rounded-3xl p-6 shadow-sm border border-slate-200 space-y-6">

                            {/* Hostname */}
                            <div className="space-y-2">
                                <label className="text-xs font-bold uppercase tracking-wider text-slate-500">Target Hostname</label>
                                <div className="relative group">
                                    <Globe className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                                    <input
                                        type="text"
                                        placeholder="e.g. smtp.gmail.com"
                                        value={host}
                                        onChange={(e) => setHost(e.target.value)}
                                        className="w-full h-12 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-xl font-mono text-sm font-bold text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all"
                                    />
                                </div>
                            </div>

                            {/* Port Selection */}
                            <div className="space-y-2">
                                <label className="text-xs font-bold uppercase tracking-wider text-slate-500">Port</label>
                                <div className="grid grid-cols-3 gap-2">
                                    {[25, 465, 587].map((p) => (
                                        <button
                                            key={p}
                                            type="button"
                                            onClick={() => setPort(p)}
                                            className={`h-10 rounded-xl font-bold text-sm transition-all border
                                                ${port === p
                                                    ? "bg-blue-600 text-white border-blue-600 shadow-lg shadow-blue-600/20"
                                                    : "bg-white text-slate-600 border-slate-200 hover:border-slate-300 hover:bg-slate-50"}
                                            `}
                                        >
                                            {p}
                                        </button>
                                    ))}
                                </div>
                            </div>

                            {/* Encryption */}
                            <div className="space-y-2">
                                <label className="text-xs font-bold uppercase tracking-wider text-slate-500">Encryption Method</label>
                                <div className="space-y-2">
                                    {['none', 'starttls', 'ssl'].map((enc) => (
                                        <label key={enc} className={`flex items-center gap-3 p-3 rounded-xl border cursor-pointer transition-all
                                            ${encryption === enc ? 'bg-blue-50 border-blue-200 ring-1 ring-blue-500/20' : 'bg-white border-slate-200 hover:bg-slate-50'}
                                        `}>
                                            <div className={`w-5 h-5 rounded-full border flex items-center justify-center
                                                ${encryption === enc ? 'border-blue-500 bg-blue-500' : 'border-slate-300'}
                                            `}>
                                                {encryption === enc && <div className="w-2 h-2 bg-white rounded-full" />}
                                            </div>
                                            <input
                                                type="radio"
                                                name="encryption"
                                                value={enc}
                                                checked={encryption === enc}
                                                onChange={() => setEncryption(enc)}
                                                className="hidden"
                                            />
                                            <div className="flex flex-col">
                                                <span className="font-bold text-sm uppercase">{enc}</span>
                                                <span className="text-xs text-slate-400 font-medium">
                                                    {enc === 'none' ? 'Plain text (Unsafe)' : enc === 'starttls' ? 'Upgrade to TLS (Recommended)' : 'Implicit SSL/TLS'}
                                                </span>
                                            </div>
                                        </label>
                                    ))}
                                </div>
                            </div>

                            <button
                                type="submit"
                                disabled={loading || !host}
                                className={`w-full h-12 rounded-xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95
                                    ${loading ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                                `}
                            >
                                {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : <Play className="w-5 h-5 fill-current" />}
                                {loading ? "Connecting..." : "Start Test"}
                            </button>
                        </form>
                    </motion.div>

                    {/* Right: Console */}
                    <motion.div
                        initial={{ opacity: 0, x: 10 }}
                        animate={{ opacity: 1, x: 0 }}
                        className="flex-1"
                    >
                        <SMTPSessionConsole
                            session={session}
                            loading={loading}
                            tlsEstablished={tlsEstablished}
                            banner={banner}
                        />
                    </motion.div>

                </div>
            </main>


        </div>
    );
}
