"use client";

import React, { useRef, useEffect } from "react";
import { motion } from "framer-motion";
import { Server, Shield, Clock, Terminal, CheckCircle, AlertTriangle, ArrowRight, ArrowLeft } from "lucide-react";

export interface SMTPSessionStep {
    step: string;
    command: string; // Formatting: "C: EHLO..."
    response: string; // Formatting: "S: 250-..."
    code: number;
    status: 'success' | 'error' | 'pending';
    timestamp?: number;
}

interface SMTPSessionConsoleProps {
    session: SMTPSessionStep[];
    loading: boolean;
    tlsEstablished: boolean;
    banner: string | null;
}

export const SMTPSessionConsole: React.FC<SMTPSessionConsoleProps> = ({ session, loading, tlsEstablished, banner }) => {
    const scrollRef = useRef<HTMLDivElement>(null);

    // Auto-scroll to bottom of log
    useEffect(() => {
        if (scrollRef.current) {
            scrollRef.current.scrollTop = scrollRef.current.scrollHeight;
        }
    }, [session]);

    const getStatusColor = (code: number, status: string) => {
        if (status === 'error') return "bg-red-100 text-red-700 border-red-200";
        if (code >= 200 && code < 400) return "bg-emerald-100 text-emerald-700 border-emerald-200";
        if (code >= 400) return "bg-amber-100 text-amber-700 border-amber-200";
        return "bg-slate-100 text-slate-700 border-slate-200";
    };

    return (
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 h-full">

            {/* Main Console (2 Cols) */}
            <div className="lg:col-span-2 flex flex-col h-[500px] bg-white border border-slate-200 rounded-3xl shadow-sm overflow-hidden">
                <div className="p-4 border-b border-slate-100 bg-slate-50 flex items-center gap-2">
                    <Terminal className="w-5 h-5 text-slate-400" />
                    <span className="font-mono text-xs font-bold uppercase tracking-wider text-slate-500">Live Session Log</span>
                </div>

                <div ref={scrollRef} className="flex-1 overflow-y-auto p-4 space-y-3 font-mono text-sm relative">
                    {session.length === 0 && !loading && (
                        <div className="absolute inset-0 flex items-center justify-center text-slate-400 opacity-50">
                            <p>Waiting to connect...</p>
                        </div>
                    )}

                    {session.map((item, i) => (
                        <motion.div
                            key={i}
                            initial={{ opacity: 0, x: -10 }}
                            animate={{ opacity: 1, x: 0 }}
                            className="flex flex-col gap-1 text-xs md:text-sm"
                        >
                            {/* Client Command */}
                            {item.command && (
                                <div className="flex items-start gap-3 self-end md:ml-12 opacity-80">
                                    <div className="bg-blue-50 text-blue-700 px-3 py-1.5 rounded-lg border border-blue-100 flex gap-2 items-center">
                                        <ArrowRight className="w-3 h-3" />
                                        <span className="font-bold">{item.step}</span>
                                        <span className="border-l border-blue-200 pl-2">{item.command}</span>
                                    </div>
                                </div>
                            )}

                            {/* Server Response */}
                            {item.response && (
                                <div className="flex items-start gap-3 self-start md:mr-12 w-full">
                                    <div className={`px-2 py-1 rounded text-[10px] font-bold border ${getStatusColor(item.code, item.status)} h-fit mt-0.5 min-w-[3rem] text-center`}>
                                        {item.code || "---"}
                                    </div>
                                    <div className="text-slate-800 break-all bg-slate-50 p-2 rounded-lg border border-slate-100 w-full">
                                        {item.response}
                                    </div>
                                </div>
                            )}
                        </motion.div>
                    ))}

                    {loading && (
                        <motion.div
                            initial={{ opacity: 0 }}
                            animate={{ opacity: 1 }}
                            className="flex items-center gap-2 text-slate-400 italic text-xs py-2"
                        >
                            <span className="w-2 h-2 bg-blue-500 rounded-full animate-pulse" />
                            Waiting for server response...
                        </motion.div>
                    )}
                </div>
            </div>

            {/* Sidebar Stats (1 Col) */}
            <div className="space-y-4">
                {/* Status Card */}
                <div className="bg-slate-900 text-white p-6 rounded-3xl shadow-lg shadow-blue-900/10">
                    <h3 className="font-bold text-slate-400 text-xs uppercase tracking-widest mb-6">Connection Status</h3>

                    <div className="space-y-6">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-3">
                                <div className={`p-2 rounded-lg ${tlsEstablished ? 'bg-emerald-500/20 text-emerald-400' : 'bg-slate-800 text-slate-500'}`}>
                                    <Shield className="w-5 h-5" />
                                </div>
                                <div>
                                    <p className="text-sm font-bold">TLS Encryption</p>
                                    <p className="text-xs text-slate-400">{tlsEstablished ? "Secure (Handshake OK)" : "Not Established"}</p>
                                </div>
                            </div>
                            {tlsEstablished && <CheckCircle className="w-5 h-5 text-emerald-500" />}
                        </div>

                        <div className="flex items-center justify-between">
                            <div className="flex items-center gap-3">
                                <div className="p-2 bg-slate-800 rounded-lg text-blue-400">
                                    <Server className="w-5 h-5" />
                                </div>
                                <div className="flex-1">
                                    <p className="text-sm font-bold">Server Logic</p>
                                    <p className="text-xs text-slate-400">{banner ? "Banner Received" : "Waiting..."}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Banner Card */}
                <div className="bg-white border border-slate-200 p-6 rounded-3xl shadow-sm">
                    <h3 className="font-bold text-slate-400 text-xs uppercase tracking-widest mb-3">Server Banner</h3>
                    <div className="bg-slate-50 p-3 rounded-xl border border-slate-100">
                        <code className="text-xs font-mono text-slate-600 break-all">
                            {banner || "No banner detected yet."}
                        </code>
                    </div>
                </div>

            </div>
        </div>
    );
};
