"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Search, Loader2, Globe, Shield, Activity, Fingerprint } from "lucide-react";

import { SPFVisualizer, SPFRecord } from "@/components/spf/SPFVisualizer";

export default function SPFClient() {
    const [domain, setDomain] = useState("");
    const [loading, setLoading] = useState(false);
    const [record, setRecord] = useState<SPFRecord | null>(null);
    const [error, setError] = useState<string | null>(null);

    const handleSearch = async (e?: React.FormEvent) => {
        if (e) e.preventDefault();
        if (!domain) return;

        setLoading(true);
        setError(null);
        setRecord(null);

        try {
            const response = await fetch('/api/dns/spf', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ domain: domain.trim() }),
            });

            const data = await response.json();

            if (!response.ok) {
                throw new Error(data.error || "Lookup failed");
            }

            setRecord(data);
        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    const getPolicy = (rec: SPFRecord) => {
        const last = rec.mechanisms[rec.mechanisms.length - 1];
        if (last && last.type === 'all') {
            switch (last.prefix) {
                case '-': return "Hard Fail (Secure)";
                case '~': return "Soft Fail (Creating Transition)";
                case '?': return "Neutral (Open)";
                case '+': return "Allow All (Insecure)";
            }
        }
        return "Implied Neutral";
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Fingerprint className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                SPF Record <span className="text-blue-600">Authority</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Validate, parse, and audit Sender Policy Framework records to ensure maximum email deliverability.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Search Box */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <form onSubmit={handleSearch} className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1 relative group">
                            <Globe className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                            <input
                                type="text"
                                placeholder="Enter Domain (e.g., google.com)"
                                value={domain}
                                onChange={(e) => setDomain(e.target.value)}
                                className="w-full h-14 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-2xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                            />
                        </div>

                        <button
                            type="submit"
                            disabled={loading || !domain}
                            className={`h-14 px-8 rounded-2xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95
                                ${loading ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                            `}
                        >
                            {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : <Search className="w-5 h-5 stroke-[3]" />}
                            {loading ? "Validating..." : "Validate"}
                        </button>
                    </form>
                </div>

                {record && (
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        className="space-y-8"
                    >
                        {/* Scoreboard Cards */}
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">

                            {/* Card 1: Status */}
                            <div className="bg-white p-6 rounded-3xl border border-slate-200 shadow-sm flex items-center gap-4">
                                <div className={`p-3 rounded-xl ${record.isValid ? 'bg-emerald-50 text-emerald-600' : 'bg-red-50 text-red-600'}`}>
                                    <Shield className="w-6 h-6" />
                                </div>
                                <div>
                                    <p className="text-xs font-bold uppercase text-slate-400 tracking-wider">Validation</p>
                                    <p className={`text-lg font-black ${record.isValid ? 'text-emerald-900' : 'text-red-900'}`}>
                                        {record.isValid ? "Valid Record" : "Error Detected"}
                                    </p>
                                </div>
                            </div>

                            {/* Card 2: Lookups */}
                            <div className="bg-white p-6 rounded-3xl border border-slate-200 shadow-sm flex items-center gap-4">
                                <div className="p-3 rounded-xl bg-blue-50 text-blue-600">
                                    <Activity className="w-6 h-6" />
                                </div>
                                <div>
                                    <p className="text-xs font-bold uppercase text-slate-400 tracking-wider">DNS Lookups</p>
                                    <p className={`text-lg font-black ${record.lookupCount > 10 ? 'text-red-600' : 'text-slate-900'}`}>
                                        {record.lookupCount} <span className="text-slate-400 text-sm font-bold">/ 10</span>
                                    </p>
                                </div>
                            </div>

                            {/* Card 3: Policy */}
                            <div className="bg-white p-6 rounded-3xl border border-slate-200 shadow-sm flex items-center gap-4">
                                <div className="p-3 rounded-xl bg-slate-100 text-slate-600">
                                    <Fingerprint className="w-6 h-6" />
                                </div>
                                <div>
                                    <p className="text-xs font-bold uppercase text-slate-400 tracking-wider">Final Policy</p>
                                    <p className="text-lg font-black text-slate-900">
                                        {getPolicy(record)}
                                    </p>
                                </div>
                            </div>

                        </div>

                        {/* Detailed Visualizer */}
                        <SPFVisualizer record={record} />
                    </motion.div>
                )}

                {/* Error State */}
                {error && (
                    <motion.div
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        className="bg-red-50 text-red-700 p-4 rounded-2xl flex items-center gap-3 border border-red-100"
                    >
                        <Shield className="w-5 h-5 flex-shrink-0" />
                        <span className="font-medium">{error}</span>
                    </motion.div>
                )}

            </main>


        </div>
    );
}
