"use client";

import React from "react";
import { motion } from "framer-motion";
import { Copy, Shield, CheckCircle, AlertTriangle, AlertCircle, Info } from "lucide-react";

export interface SPFMechanism {
    prefix: string; // +, -, ~, ?
    type: string;   // ip4, include, a, mx, all
    value: string;
    description: string;
    isLookup: boolean; // Counts towards 10 limit
}

export interface SPFRecord {
    raw: string;
    mechanisms: SPFMechanism[];
    lookupCount: number;
    isValid: boolean;
    errors: string[];
}

interface SPFVisualizerProps {
    record: SPFRecord;
}

export const SPFVisualizer: React.FC<SPFVisualizerProps> = ({ record }) => {

    const copyToClipboard = () => {
        navigator.clipboard.writeText(record.raw);
    };

    const getPrefixColor = (prefix: string) => {
        switch (prefix) {
            case '+': return "bg-emerald-100 text-emerald-700 border-emerald-200"; // Pass
            case '-': return "bg-red-100 text-red-700 border-red-200"; // Fail
            case '~': return "bg-amber-100 text-amber-700 border-amber-200"; // SoftFail
            case '?': return "bg-slate-100 text-slate-700 border-slate-200"; // Neutral
            default: return "bg-slate-100 text-slate-700";
        }
    };

    const getTypeDescription = (type: string) => {
        switch (type.toLowerCase()) {
            case 'ip4': return "Authorizes a specific IPv4 address or range.";
            case 'ip6': return "Authorizes a specific IPv6 address or range.";
            case 'include': return "Includes SPF record from another domain.";
            case 'a': return "Authorizes IP addresses of the domain's A record.";
            case 'mx': return "Authorizes IP addresses of the domain's MX records.";
            case 'all': return "Wildcard policy for all other senders.";
            case 'redirect': return "Redirects SPF check to another domain.";
            case 'ptr': return "Authorizes IPs via reverse DNS (Deprecated/Slow).";
            default: return "SPF Mechanism";
        }
    };

    return (
        <div className="space-y-6">

            {/* Raw Record Card */}
            <div className="bg-white border border-slate-200 rounded-3xl p-6 shadow-sm">
                <div className="flex items-center justify-between mb-4">
                    <h3 className="font-bold text-slate-900 text-sm uppercase tracking-wider flex items-center gap-2">
                        <Shield className="w-4 h-4 text-blue-600" />
                        Raw Record
                    </h3>
                    <button
                        onClick={copyToClipboard}
                        className="text-slate-400 hover:text-blue-600 transition-colors"
                        title="Copy to Clipboard"
                    >
                        <Copy className="w-5 h-5" />
                    </button>
                </div>
                <div className="bg-slate-50 p-4 rounded-xl border border-slate-100 font-mono text-sm break-all text-slate-800">
                    {record.raw}
                </div>
            </div>

            {/* Lookup Gauge & Status */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">

                {/* Lookup Counter */}
                <div className="bg-white border border-slate-200 rounded-3xl p-6 shadow-sm flex flex-col justify-center">
                    <div className="flex items-center justify-between mb-2">
                        <h3 className="font-bold text-slate-500 text-xs uppercase tracking-wider">DNS Lookups</h3>
                        <span className={`text-xs font-bold px-2 py-0.5 rounded-full ${record.lookupCount > 10 ? 'bg-red-100 text-red-600' : 'bg-slate-100 text-slate-600'}`}>
                            Limit: 10
                        </span>
                    </div>

                    <div className="relative pt-2">
                        <div className="flex items-end gap-2 mb-1">
                            <span className={`text-4xl font-black ${record.lookupCount > 10 ? 'text-red-500' : 'text-slate-900'}`}>
                                {record.lookupCount}
                            </span>
                            <span className="text-slate-400 font-bold mb-1">/ 10</span>
                        </div>

                        {/* Progress Bar */}
                        <div className="h-3 w-full bg-slate-100 rounded-full overflow-hidden">
                            <motion.div
                                initial={{ width: 0 }}
                                animate={{ width: `${Math.min(100, (record.lookupCount / 10) * 100)}%` }}
                                className={`h-full rounded-full transition-colors duration-500 ${record.lookupCount > 10 ? 'bg-red-500' :
                                        record.lookupCount > 7 ? 'bg-amber-500' : 'bg-emerald-500'
                                    }`}
                            />
                        </div>

                        <p className="text-xs text-slate-400 mt-2 font-medium">
                            {record.lookupCount > 10
                                ? "Critical: Limits exceeded. Emails may fail."
                                : "Within RFC limits. Good deliverability."}
                        </p>
                    </div>
                </div>

                {/* Validation Status */}
                <div className={`border rounded-3xl p-6 shadow-sm flex flex-col justify-center relative overflow-hidden
                    ${record.isValid ? 'bg-emerald-50 border-emerald-100' : 'bg-red-50 border-red-100'}
                `}>
                    <div className="relative z-10">
                        <div className="flex items-center gap-3 mb-2">
                            {record.isValid ? (
                                <CheckCircle className="w-6 h-6 text-emerald-600" />
                            ) : (
                                <AlertTriangle className="w-6 h-6 text-red-600" />
                            )}
                            <h3 className={`font-bold text-lg ${record.isValid ? 'text-emerald-900' : 'text-red-900'}`}>
                                {record.isValid ? "Valid SPF Record" : "Invalid Syntax"}
                            </h3>
                        </div>

                        <div className="space-y-1">
                            {record.errors.length > 0 ? (
                                record.errors.map((err, i) => (
                                    <p key={i} className="text-sm font-medium text-red-700 flex items-center gap-2">
                                        <AlertCircle className="w-3 h-3" /> {err}
                                    </p>
                                ))
                            ) : (
                                <p className="text-emerald-800 text-sm font-medium">
                                    Structure matches valid syntax requirements.
                                </p>
                            )}
                        </div>
                    </div>
                </div>
            </div>

            {/* Mechanisms Table */}
            <div className="bg-white border border-slate-200 rounded-3xl overflow-hidden shadow-sm">
                <div className="p-4 bg-slate-50 border-b border-slate-200 flex items-center gap-2">
                    <Info className="w-4 h-4 text-slate-400" />
                    <span className="font-bold text-xs uppercase tracking-wider text-slate-500">Mechanism Breakdown</span>
                </div>

                <div className="divide-y divide-slate-100">
                    {record.mechanisms.map((mech, i) => (
                        <div key={i} className="p-4 flex flex-col md:flex-row gap-4 md:items-center hover:bg-slate-50/50 transition-colors">
                            {/* Visual Tag */}
                            <div className="flex-shrink-0 min-w-[140px] flex items-center gap-2 font-mono text-sm">
                                <span className={`px-2 py-0.5 rounded border text-xs font-bold ${getPrefixColor(mech.prefix)}`}>
                                    {mech.prefix || '+'}
                                </span>
                                <span className="font-bold text-slate-700">{mech.type}</span>
                            </div>

                            {/* Value */}
                            <div className="flex-1 font-mono text-sm text-slate-600 break-all">
                                {mech.value || '-'}
                            </div>

                            {/* Description */}
                            <div className="md:w-1/3 text-xs text-slate-500 font-medium">
                                {mech.description || getTypeDescription(mech.type)}
                            </div>

                            {/* Cost Badge */}
                            <div className="flex-shrink-0">
                                {mech.isLookup && (
                                    <span className="text-[10px] font-bold uppercase bg-blue-50 text-blue-600 px-2 py-1 rounded border border-blue-100">
                                        +1 Lookup
                                    </span>
                                )}
                            </div>
                        </div>
                    ))}
                </div>
            </div>

        </div>
    );
};
