"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Search, Loader2, ShieldCheck, ShieldAlert, Lock } from "lucide-react";

import { SSLDetails, SSLData } from "@/components/ssl-checker/SSLDetails";

export default function SSLCheckerClient() {
    const [domain, setDomain] = useState("");
    const [loading, setLoading] = useState(false);
    const [data, setData] = useState<SSLData | null>(null);
    const [error, setError] = useState("");

    const handleCheck = async () => {
        if (!domain) return;
        setLoading(true);
        setError("");
        setData(null);

        try {
            const res = await fetch('/api/ssl', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ domain })
            });
            const json = await res.json();

            if (!res.ok) {
                throw new Error(json.error || "Failed to check SSL");
            }
            setData(json);
        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-white flex flex-col font-sans text-slate-900">
            {/* Header */}
            <section className="bg-[#F8FAFC] border-b border-[#E2E8F0] pt-20 pb-16 relative overflow-hidden">
                <div className="absolute inset-0 opacity-5 pointer-events-none" style={{
                    backgroundImage: `radial-gradient(circle at 2px 2px, #94a3b8 1px, transparent 0)`,
                    backgroundSize: '24px 24px'
                }} />

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center relative z-10">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                    >
                        <div className="inline-flex items-center justify-center p-3 bg-blue-50 rounded-2xl mb-6 text-[#2563EB]">
                            <Lock className="w-8 h-8" />
                        </div>
                        <h1 className="text-4xl md:text-5xl font-extrabold text-[#0F172A] mb-4 font-jakarta">
                            SSL Certificate Verifier
                        </h1>
                        <p className="text-lg text-[#475569] max-w-2xl mx-auto mb-10">
                            Deep-scan your website’s SSL configuration, chain of trust, and security vulnerabilities.
                        </p>

                        {/* Search Bar */}
                        <div className="max-w-xl mx-auto bg-white p-2 rounded-2xl shadow-lg border border-[#E2E8F0] flex items-center">
                            <input
                                type="text"
                                placeholder="Enter domain (e.g. google.com)"
                                value={domain}
                                onChange={(e) => setDomain(e.target.value)}
                                onKeyDown={(e) => e.key === 'Enter' && handleCheck()}
                                className="flex-1 h-12 pl-4 pr-4 bg-transparent text-lg font-medium text-[#0F172A] placeholder-slate-400 outline-none"
                                suppressHydrationWarning
                            />
                            <button
                                onClick={handleCheck}
                                disabled={loading || !domain}
                                className="h-12 px-8 bg-[#2563EB] hover:bg-[#1E3A8A] text-white font-bold rounded-xl transition-all shadow-md active:scale-95 disabled:opacity-70 flex items-center justify-center gap-2"
                            >
                                {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : "Verify"}
                            </button>
                        </div>
                    </motion.div>
                </div>
            </section>

            {/* Error Message */}
            {error && (
                <div className="max-w-3xl mx-auto px-4 pt-8">
                    <div className="bg-red-50 border border-red-100 rounded-xl p-4 flex items-center gap-3 text-red-700">
                        <ShieldAlert className="w-6 h-6 shrink-0" />
                        <span className="font-medium">{error}</span>
                    </div>
                </div>
            )}

            {/* Results */}
            {data && (
                <main className="flex-1 max-w-5xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full">

                    {/* Status Hero */}
                    <div className="mb-10 text-center">
                        <div className={`inline-flex items-center gap-3 px-6 py-3 rounded-full border mb-4
                            ${data.authorized ? "bg-green-50 border-green-200 text-green-700" : "bg-red-50 border-red-200 text-red-700"}
                        `}>
                            {data.authorized ? <ShieldCheck className="w-6 h-6" /> : <ShieldAlert className="w-6 h-6" />}
                            <span className="text-xl font-bold">
                                {data.authorized ? "Secure Connection Verified" : "Certificate Issues Detected"}
                            </span>
                        </div>
                        <h2 className="text-3xl font-bold text-slate-800">{data.subject}</h2>
                    </div>

                    <SSLDetails data={data} />
                </main>
            )}


        </div>
    );
}
