"use client";

import React, { useState } from "react";
import { Copy, Download, ExternalLink, Search, CheckCircle, XCircle, AlertCircle } from "lucide-react";

export interface SubdomainEntry {
    subdomain: string;
    ip: string | null;
    provider: string | null;
    lastSeen: string;
    isReachable?: boolean;
}

interface SubdomainListProps {
    data: SubdomainEntry[];
}

export const SubdomainList: React.FC<SubdomainListProps> = ({ data }) => {
    const [search, setSearch] = useState("");
    const [page, setPage] = useState(1);
    const ITEMS_PER_PAGE = 20;

    const filtered = data.filter(item =>
        item.subdomain.toLowerCase().includes(search.toLowerCase()) ||
        (item.ip && item.ip.includes(search))
    );

    const totalPages = Math.ceil(filtered.length / ITEMS_PER_PAGE);
    const paginated = filtered.slice((page - 1) * ITEMS_PER_PAGE, page * ITEMS_PER_PAGE);

    const handleDownload = (format: 'csv' | 'json') => {
        if (format === 'json') {
            const blob = new Blob([JSON.stringify(filtered, null, 2)], { type: 'application/json' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `subdomains-${new Date().toISOString().slice(0, 10)}.json`;
            a.click();
        } else {
            const csv = [
                ['Subdomain', 'IP Address', 'Provider', 'Last Seen'],
                ...filtered.map(row => [row.subdomain, row.ip || 'N/A', row.provider || 'N/A', row.lastSeen])
            ].map(e => e.join(',')).join('\n');

            const blob = new Blob([csv], { type: 'text/csv' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `subdomains-${new Date().toISOString().slice(0, 10)}.csv`;
            a.click();
        }
    };

    return (
        <div className="space-y-4">
            {/* Toolbar */}
            <div className="flex flex-col md:flex-row gap-4 justify-between items-center bg-white p-4 rounded-xl border border-slate-200 shadow-sm">
                <div className="relative w-full md:w-80">
                    <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-slate-400" />
                    <input
                        type="text"
                        placeholder="Filter subdomains..."
                        value={search}
                        onChange={(e) => { setSearch(e.target.value); setPage(1); }}
                        className="w-full pl-10 pr-4 py-2 bg-slate-50 border border-slate-200 rounded-lg text-sm font-medium outline-none focus:ring-2 focus:ring-blue-500/20 focus:border-blue-500"
                    />
                </div>
                <div className="flex gap-2">
                    <button
                        onClick={() => handleDownload('csv')}
                        className="px-4 py-2 text-sm font-bold text-slate-600 hover:text-blue-600 bg-slate-50 hover:bg-blue-50 rounded-lg border border-slate-200 transition-colors flex items-center gap-2"
                    >
                        <Download className="w-4 h-4" /> CSV
                    </button>
                    <button
                        onClick={() => handleDownload('json')}
                        className="px-4 py-2 text-sm font-bold text-slate-600 hover:text-blue-600 bg-slate-50 hover:bg-blue-50 rounded-lg border border-slate-200 transition-colors flex items-center gap-2"
                    >
                        <Download className="w-4 h-4" /> JSON
                    </button>
                </div>
            </div>

            {/* Table */}
            <div className="bg-white border border-slate-200 rounded-xl overflow-hidden shadow-sm">
                <div className="overflow-x-auto">
                    <table className="w-full text-left text-sm">
                        <thead className="bg-slate-50 border-b border-slate-200 text-slate-500 font-bold uppercase tracking-wider text-xs">
                            <tr>
                                <th className="px-6 py-4">Subdomain</th>
                                <th className="px-6 py-4">IP Address</th>
                                <th className="px-6 py-4">Seen On</th>
                                <th className="px-6 py-4 text-right">Action</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-slate-100">
                            {paginated.length > 0 ? paginated.map((item, idx) => (
                                <tr key={idx} className="hover:bg-slate-50/80 transition-colors">
                                    <td className="px-6 py-3 font-mono font-medium text-slate-700 select-all">
                                        {item.subdomain}
                                    </td>
                                    <td className="px-6 py-3">
                                        {item.ip ? (
                                            <span className="inline-flex items-center gap-2 px-2.5 py-1 rounded-full bg-blue-50 text-blue-700 text-xs font-bold font-mono">
                                                {item.ip}
                                                {item.provider && <span className="opacity-60 border-l border-blue-200 pl-2 ml-1">{item.provider}</span>}
                                            </span>
                                        ) : (
                                            <span className="text-slate-400 font-mono text-xs">-</span>
                                        )}
                                    </td>
                                    <td className="px-6 py-3 text-slate-500 whitespace-nowrap">
                                        {new Date(item.lastSeen).toLocaleDateString()}
                                    </td>
                                    <td className="px-6 py-3 text-right">
                                        <a
                                            href={`http://${item.subdomain}`}
                                            target="_blank"
                                            rel="noopener noreferrer"
                                            className="inline-flex items-center justify-center p-2 text-slate-400 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-colors"
                                        >
                                            <ExternalLink className="w-4 h-4" />
                                        </a>
                                    </td>
                                </tr>
                            )) : (
                                <tr>
                                    <td colSpan={4} className="px-6 py-8 text-center text-slate-500 italic">
                                        No subdomains match your search.
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>

                {/* Pagination */}
                {totalPages > 1 && (
                    <div className="bg-slate-50 border-t border-slate-200 p-4 flex items-center justify-between">
                        <span className="text-sm text-slate-500 font-medium">
                            Showing {Math.min(filtered.length, (page - 1) * ITEMS_PER_PAGE + 1)}-{Math.min(filtered.length, page * ITEMS_PER_PAGE)} of {filtered.length}
                        </span>
                        <div className="flex gap-2">
                            <button
                                onClick={() => setPage(p => Math.max(1, p - 1))}
                                disabled={page === 1}
                                className="px-3 py-1 rounded-md border border-slate-200 bg-white text-sm font-medium text-slate-600 disabled:opacity-50 hover:bg-slate-50"
                            >
                                Previous
                            </button>
                            <button
                                onClick={() => setPage(p => Math.min(totalPages, p + 1))}
                                disabled={page === totalPages}
                                className="px-3 py-1 rounded-md border border-slate-200 bg-white text-sm font-medium text-slate-600 disabled:opacity-50 hover:bg-slate-50"
                            >
                                Next
                            </button>
                        </div>
                    </div>
                )}
            </div>
        </div>
    );
};
