"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { Network, Play, Loader2, Globe } from "lucide-react";

import { TraceResults, HopData } from "@/components/trace/TraceResults";

export default function TraceClient() {
    const [host, setHost] = useState("");
    const [maxHops, setMaxHops] = useState(20);
    const [isTracing, setIsTracing] = useState(false);
    const [hops, setHops] = useState<HopData[]>([]);
    const [rawOutput, setRawOutput] = useState<string[]>([]);

    const handleTrace = async () => {
        if (!host) return;
        setIsTracing(true);
        setHops([]);
        setRawOutput([]);

        try {
            const response = await fetch('/api/trace', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ host, maxHops }),
            });

            if (!response.body) throw new Error("No response body");

            const reader = response.body.getReader();
            const decoder = new TextDecoder();
            let buffer = '';

            while (true) {
                const { done, value } = await reader.read();
                if (done) break;

                const chunk = decoder.decode(value, { stream: true });
                buffer += chunk;

                const lines = buffer.split(/\r?\n/);
                buffer = lines.pop() || '';

                for (const line of lines) {
                    if (!line.trim()) continue;
                    setRawOutput(prev => [...prev, line]);
                    parseLine(line);
                }
            }

        } catch (error: any) {
            setRawOutput(prev => [...prev, `Error: ${error.message}`]);
        } finally {
            setIsTracing(false);
        }
    };

    const parseLine = (line: string) => {
        // Regex attempts to be generic for Windows/Linux
        // Windows:  1     1 ms     <1 ms     1 ms  192.168.1.1
        // Linux:    1  192.168.1.1 (192.168.1.1)  0.123 ms...

        // 1. Extract Hop Number
        const hopMatch = line.match(/^\s*(\d+)\s+/);
        if (!hopMatch) return;
        const hopNum = parseInt(hopMatch[1]);

        // 2. Extract RTTs (look for "ms" or "<1 ms")
        const rttMatches = [...line.matchAll(/(?:<)?([\d.]+)\s*ms/g)];
        const rtts = rttMatches.map(m => parseFloat(m[1]));

        // 3. Extract IP/Host
        // Simple logic: look for IP-like pattern
        const ipMatch = line.match(/(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})/);
        const ip = ipMatch ? ipMatch[1] : "";

        // Host might be different, but parsing plain text "hostname" is hard without stricter regex.
        // For now, if we find an IP, use it. If not, and line says "Request timed out" or "* * *", it's a timeout.
        const isTimeout = line.includes("Request timed out") || line.includes("*        *        *");

        if (ip || isTimeout) {
            setHops(prev => {
                const existing = prev.find(h => h.hop === hopNum);
                if (existing) return prev; // Already processed this hop (unlikely in stream unless duplicate lines)

                const validRtts = rtts.filter(r => !isNaN(r));
                const sum = validRtts.reduce((a, b) => a + b, 0);
                const avg = validRtts.length ? sum / validRtts.length : 0;
                const best = validRtts.length ? Math.min(...validRtts) : 9999;
                const worst = validRtts.length ? Math.max(...validRtts) : 0;

                // StDev
                const variance = validRtts.length ? validRtts.reduce((a, b) => a + Math.pow(b - avg, 2), 0) / validRtts.length : 0;
                const stdev = Math.sqrt(variance);

                const newHop: HopData = {
                    hop: hopNum,
                    host: ip || "Unknown", // Ideally parse hostname if present
                    ip: ip || "*",
                    rtt: validRtts,
                    loss: isTimeout ? 100 : 0, // Simplified loss for tracert (it's usually 0 or 100 per line)
                    avg,
                    best,
                    worst,
                    stdev,
                    status: isTimeout ? 'timeout' : 'done' // Tracert lines are "final" when printed usually
                };
                return [...prev, newHop];
            });
        }
    };

    return (
        <div className="min-h-screen bg-[#F8FAFC] flex flex-col font-sans text-slate-900 selection:bg-blue-100 selection:text-blue-900">

            {/* Header */}
            <div className="bg-white border-b border-slate-200">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10">
                    <motion.div initial={{ opacity: 0, y: 10 }} animate={{ opacity: 1, y: 0 }}>
                        <div className="flex items-center gap-3 mb-2">
                            <div className="p-2 bg-blue-600 rounded-lg shadow-lg shadow-blue-600/20 text-white">
                                <Network className="w-6 h-6" />
                            </div>
                            <h1 className="text-2xl font-black text-slate-900 tracking-tight font-jakarta">
                                Network Path <span className="text-blue-600">Trace</span>
                            </h1>
                        </div>
                        <p className="text-slate-500 font-medium max-w-2xl">
                            Visualize the hop-by-hop journey of packets to any global destination.
                        </p>
                    </motion.div>
                </div>
            </div>

            {/* Main Content */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-10 w-full space-y-8">

                {/* Controls */}
                <div className="bg-white rounded-3xl p-6 shadow-xl shadow-blue-900/5 border border-slate-200">
                    <div className="flex flex-col md:flex-row gap-4">
                        <div className="flex-1 relative group">
                            <Globe className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-blue-500 transition-colors" />
                            <input
                                type="text"
                                placeholder="Enter Host (e.g., google.com or 1.1.1.1)"
                                value={host}
                                onChange={(e) => setHost(e.target.value)}
                                className="w-full h-14 pl-12 pr-4 bg-slate-50 border border-slate-200 rounded-2xl font-mono text-slate-900 outline-none focus:bg-white focus:border-blue-500 focus:ring-4 focus:ring-blue-500/10 transition-all font-bold"
                            />
                        </div>

                        <div className="w-full md:w-48 relative flex items-center bg-slate-50 border border-slate-200 rounded-2xl px-4 h-14">
                            <span className="text-xs font-bold text-slate-400 uppercase mr-2">Max Hops:</span>
                            <input
                                type="range"
                                min="5"
                                max="30"
                                value={maxHops}
                                onChange={(e) => setMaxHops(Number(e.target.value))}
                                className="flex-1 accent-blue-600 cursor-pointer"
                            />
                            <span className="ml-3 font-bold text-slate-700 w-6 text-center">{maxHops}</span>
                        </div>

                        <button
                            onClick={handleTrace}
                            disabled={isTracing || !host}
                            className={`h-14 px-8 rounded-2xl font-bold flex items-center justify-center gap-2 transition-all shadow-lg active:scale-95
                                ${isTracing ? "bg-slate-100 text-slate-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700 text-white shadow-blue-600/20"}
                            `}
                        >
                            {isTracing ? <Loader2 className="w-5 h-5 animate-spin" /> : <Play className="w-5 h-5 fill-current" />}
                            {isTracing ? "Tracing..." : "Launch Trace"}
                        </button>
                    </div>
                </div>

                {/* Results Component */}
                <TraceResults hops={hops} rawOutput={rawOutput} isTracing={isTracing} />

            </main>


        </div>
    );
}
