"use client";

import React, { useState } from "react";
import { motion, AnimatePresence } from "framer-motion";
import { Copy, Check, ChevronDown, ChevronUp, Shield, Mail, Key, FileText } from "lucide-react";

export interface TXTRecordObj {
    value: string;
    type: 'SPF' | 'DMARC' | 'Verification' | 'General';
}

interface TXTRecordListProps {
    records: TXTRecordObj[];
}

export const TXTRecordList: React.FC<TXTRecordListProps> = ({ records }) => {

    // Copy state helper
    const [copiedIndex, setCopiedIndex] = useState<number | null>(null);

    const handleCopy = (text: string, index: number) => {
        navigator.clipboard.writeText(text);
        setCopiedIndex(index);
        setTimeout(() => setCopiedIndex(null), 2000);
    };

    return (
        <div className="space-y-4">
            {records.map((record, index) => (
                <RecordCard
                    key={index}
                    record={record}
                    index={index}
                    isCopied={copiedIndex === index}
                    onCopy={() => handleCopy(record.value, index)}
                />
            ))}
        </div>
    );
};

const RecordCard: React.FC<{
    record: TXTRecordObj,
    index: number,
    isCopied: boolean,
    onCopy: () => void
}> = ({ record, index, isCopied, onCopy }) => {

    const [expanded, setExpanded] = useState(false);

    const getTypeStyles = (type: string) => {
        switch (type) {
            case 'SPF': return "bg-blue-100 text-blue-700 border-blue-200";
            case 'DMARC': return "bg-purple-100 text-purple-700 border-purple-200";
            case 'Verification': return "bg-green-100 text-green-700 border-green-200";
            default: return "bg-slate-100 text-slate-600 border-slate-200";
        }
    };

    const getTypeIcon = (type: string) => {
        switch (type) {
            case 'SPF': return <Shield className="w-3 h-3" />;
            case 'DMARC': return <Mail className="w-3 h-3" />;
            case 'Verification': return <Key className="w-3 h-3" />;
            default: return <FileText className="w-3 h-3" />;
        }
    };

    // Parse SPF parts if SPF
    const isSPF = record.type === 'SPF';
    const spfParts = isSPF ? record.value.split(' ').filter(p => !!p).slice(1) : []; // skip v=spf1

    return (
        <motion.div
            initial={{ opacity: 0, y: 10 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: index * 0.05 }}
            className="bg-white border border-slate-200 rounded-xl p-0 hover:shadow-lg hover:shadow-blue-500/5 transition-all group overflow-hidden"
        >
            <div className="p-4 flex items-start gap-4">
                {/* Icon/Badge */}
                <div className={`shrink-0 mt-1 inline-flex items-center gap-1.5 px-2.5 py-1 rounded-full text-xs font-bold uppercase border ${getTypeStyles(record.type)}`}>
                    {getTypeIcon(record.type)}
                    {record.type}
                </div>

                {/* Content */}
                <div className="flex-1 min-w-0">
                    <div className="font-mono text-sm text-slate-700 break-all leading-relaxed bg-slate-50 p-3 rounded-lg border border-slate-100">
                        {record.value}
                    </div>

                    {/* SPF Parser Toggle */}
                    {isSPF && (
                        <button
                            onClick={() => setExpanded(!expanded)}
                            className="mt-2 text-xs font-semibold text-blue-600 hover:text-blue-700 flex items-center gap-1 transition-colors"
                        >
                            {expanded ? "Hide Breakdown" : "Explain SPF"}
                            {expanded ? <ChevronUp className="w-3 h-3" /> : <ChevronDown className="w-3 h-3" />}
                        </button>
                    )}
                </div>

                {/* Copy Button */}
                <button
                    onClick={onCopy}
                    className="p-2 text-slate-400 hover:text-blue-600 bg-transparent hover:bg-blue-50 rounded-lg transition-all"
                    title="Copy to clipboard"
                >
                    {isCopied ? <Check className="w-4 h-4 text-green-500" /> : <Copy className="w-4 h-4" />}
                </button>
            </div>

            {/* SPF Breakdown Panel */}
            <AnimatePresence>
                {expanded && isSPF && (
                    <motion.div
                        initial={{ height: 0, opacity: 0 }}
                        animate={{ height: "auto", opacity: 1 }}
                        exit={{ height: 0, opacity: 0 }}
                        className="bg-blue-50/50 border-t border-blue-100 px-4 py-3"
                    >
                        <div className="text-xs font-bold text-blue-800 mb-2 uppercase tracking-wide">Mechanism Breakdown</div>
                        <div className="grid grid-cols-1 sm:grid-cols-2 gap-2 text-xs">
                            <div className="flex gap-2 items-center bg-white p-2 rounded border border-blue-100">
                                <span className="font-mono font-bold text-blue-600">v=spf1</span>
                                <span className="text-slate-600">Identify as SPF record version 1</span>
                            </div>
                            {spfParts.map((part, i) => {
                                let label = "Unknown Mechanism";
                                if (part.startsWith("include:")) label = "Authorize external domain's IPs";
                                if (part.startsWith("ip4:")) label = "Authorize specific IPv4 address/range";
                                if (part.startsWith("ip6:")) label = "Authorize specific IPv6 address/range";
                                if (part.startsWith("a")) label = "Authorize domain's A record IP";
                                if (part.startsWith("mx")) label = "Authorize domain's MX record IPs";
                                if (part === "~all") label = "Soft Fail (Mark unauth email as spam)";
                                if (part === "-all") label = "Hard Fail (Reject unauth email)";
                                if (part === "+all") label = "Allow all (Insecure)";

                                return (
                                    <div key={i} className="flex gap-2 items-center bg-white p-2 rounded border border-blue-100">
                                        <span className="font-mono font-bold text-slate-800 max-w-[120px] truncate" title={part}>{part}</span>
                                        <span className="text-slate-600">{label}</span>
                                    </div>
                                );
                            })}
                        </div>
                    </motion.div>
                )}
            </AnimatePresence>
        </motion.div>
    );
};
