"use client";

import React, { useState } from "react";
import { motion } from "framer-motion";
import { FileText, Search, Loader2, Globe } from "lucide-react";

import { TXTRecordList, TXTRecordObj } from "@/components/txt-record/TXTRecordList";

export default function TxtRecordClient() {
    const [domain, setDomain] = useState("");
    const [loading, setLoading] = useState(false);
    const [records, setRecords] = useState<TXTRecordObj[]>([]);
    const [error, setError] = useState("");
    const [searched, setSearched] = useState(false);

    const handleSearch = async (e?: React.FormEvent) => {
        if (e) e.preventDefault();
        if (!domain) return;
        setLoading(true);
        setError("");
        setRecords([]);
        setSearched(true);

        try {
            const res = await fetch('/api/dns/txt', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ domain })
            });
            const json = await res.json();

            if (!res.ok) {
                throw new Error(json.error || "Failed to fetch records");
            }

            // Categorize
            const categorized = json.records.map((val: string) => {
                let type: TXTRecordObj['type'] = 'General';
                if (val.startsWith('v=spf1')) type = 'SPF';
                else if (val.startsWith('v=DMARC1')) type = 'DMARC';
                else if (val.includes('verification') || val.includes('validate')) type = 'Verification';

                return { value: val, type };
            });

            // Sort: SPF/DMARC first
            categorized.sort((a: TXTRecordObj, b: TXTRecordObj) => {
                const score = (t: string) => t === 'SPF' ? 3 : t === 'DMARC' ? 2 : t === 'Verification' ? 1 : 0;
                return score(b.type) - score(a.type);
            });

            setRecords(categorized);

        } catch (err: any) {
            setError(err.message);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-white flex flex-col font-sans text-slate-900">
            {/* Header */}
            <section className="bg-[#F8FAFC] border-b border-[#E2E8F0] pt-20 pb-12">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                    >
                        <div className="inline-flex items-center justify-center p-3 bg-blue-50 rounded-2xl mb-6 text-[#2563EB]">
                            <FileText className="w-8 h-8" />
                        </div>
                        <h1 className="text-4xl md:text-5xl font-extrabold text-[#0F172A] mb-4 font-jakarta">
                            TXT Record Lookup
                        </h1>
                        <p className="text-lg text-[#475569] max-w-2xl mx-auto">
                            Retrieve and analyze all TXT records for a domain, including SPF, DKIM, and site verification codes.
                        </p>
                    </motion.div>
                </div>
            </section>

            {/* Search */}
            <section className="max-w-3xl mx-auto px-4 -mt-8 relative z-10 w-full mb-12">
                <form onSubmit={handleSearch} className="bg-white p-2 rounded-2xl shadow-xl shadow-slate-200/50 border border-[#E2E8F0] flex items-center gap-2">
                    <Globe className="text-slate-400 ml-4 w-5 h-5 flex-shrink-0" />
                    <input
                        type="text"
                        placeholder="Enter domain (e.g. google.com)"
                        value={domain}
                        onChange={(e) => setDomain(e.target.value)}
                        className="flex-1 h-12 bg-transparent text-lg font-medium text-[#0F172A] placeholder-slate-400 outline-none"
                        suppressHydrationWarning
                    />
                    <button
                        type="submit"
                        disabled={loading || !domain}
                        className="h-12 px-8 bg-[#2563EB] hover:bg-[#1E3A8A] text-white font-bold rounded-xl transition-all shadow-md active:scale-95 disabled:opacity-70 flex items-center justify-center gap-2"
                    >
                        {loading ? <Loader2 className="w-5 h-5 animate-spin" /> : "Search"}
                    </button>
                </form>
            </section>

            {/* Results */}
            <main className="flex-1 max-w-4xl mx-auto px-4 w-full pb-20">
                {error && (
                    <div className="bg-red-50 border border-red-100 rounded-xl p-4 text-center text-red-700 font-medium mb-8">
                        {error}
                    </div>
                )}

                {records.length > 0 && (
                    <>
                        <div className="flex items-center justify-between mb-6 px-2">
                            <h2 className="text-sm font-bold text-slate-500 uppercase tracking-wider">
                                Total Records Found: <span className="text-slate-900">{records.length}</span>
                            </h2>
                        </div>
                        <TXTRecordList records={records} />
                    </>
                )}

                {searched && !loading && !error && records.length === 0 && (
                    <div className="text-center py-20 text-slate-400">
                        <p>No TXT records found for this domain.</p>
                    </div>
                )}
            </main>


        </div>
    );
}
