"use client";

import React, { useState, useEffect } from "react";
import { motion } from "framer-motion";
import { Search, FileText, Loader2, AlertCircle, Clock, X, ArrowRight } from "lucide-react";

import { WhoisResult, WhoisData } from "@/components/whois/WhoisResult";

export default function WhoisClient() {
    const [domain, setDomain] = useState("");
    const [isLoading, setIsLoading] = useState(false);
    const [data, setData] = useState<WhoisData | null>(null);
    const [error, setError] = useState("");
    const [recentSearches, setRecentSearches] = useState<string[]>([]);

    useEffect(() => {
        const saved = localStorage.getItem("whois_recent");
        if (saved) {
            setRecentSearches(JSON.parse(saved));
        }
    }, []);

    const addToHistory = (dom: string) => {
        const clean = dom.toLowerCase().trim();
        const updated = [clean, ...recentSearches.filter(d => d !== clean)].slice(0, 5);
        setRecentSearches(updated);
        localStorage.setItem("whois_recent", JSON.stringify(updated));
    };

    const removeFromHistory = (e: React.MouseEvent, dom: string) => {
        e.stopPropagation();
        const updated = recentSearches.filter(d => d !== dom);
        setRecentSearches(updated);
        localStorage.setItem("whois_recent", JSON.stringify(updated));
    };

    const handleSearch = async (e?: React.FormEvent, searchDomain?: string) => {
        if (e) e.preventDefault();
        const targetDomain = searchDomain || domain;
        if (!targetDomain) return;

        // Create a 'cleaner' input for visual
        setDomain(targetDomain);

        setIsLoading(true);
        setError("");
        setData(null);

        try {
            const res = await fetch('/api/whois', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ domain: targetDomain })
            });

            if (!res.ok) {
                const errData = await res.json();
                throw new Error(errData.error || "Lookup Failed");
            }

            const result = await res.json();
            setData(result);
            addToHistory(targetDomain);
        } catch (err: any) {
            setError(err.message);
        } finally {
            setIsLoading(false);
        }
    };

    return (
        <div className="min-h-screen bg-white flex flex-col font-sans text-slate-900">
            {/* Header / Hero */}
            <section className="bg-[#F8FAFC] border-b border-[#E2E8F0] pt-20 pb-16 relative overflow-hidden">
                {/* Background Pattern */}
                <div className="absolute inset-0 opacity-10 pointer-events-none" style={{
                    backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cpath d='M54.627 0l.83.828-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM22.485 0l.83.828-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM0 22.485l.828.83-1.415 1.415-.828-.828-.828.828L-3.658 22.485l.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM0 54.627l.828.83-1.415 1.415-.828-.828-.828.828L-3.658 54.627l.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM54.627 32.142l.83.828-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM22.485 32.142l.83.828-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM32.142 54.627l.828.83-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM32.142 22.485l.828.83-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM54.627 0l.83.828-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM22.485 0l.83.828-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM0 22.485l.828.83-1.415 1.415-.828-.828-.828.828L-3.658 22.485l.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM0 54.627l.828.83-1.415 1.415-.828-.828-.828.828L-3.658 54.627l.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM54.627 32.142l.83.828-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM22.485 32.142l.83.828-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM32.142 54.627l.828.83-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828zM32.142 22.485l.828.83-1.415 1.415-.828-.828-.828.828-1.415-1.415.828-.828-.828-.828 1.415-1.415.828.828.828-.828 1.415 1.415-.828.828z' fill='%2394a3b8' fill-opacity='0.4' fill-rule='evenodd'/%3E%3C/svg%3E")`
                }} />

                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center relative z-10">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.5 }}
                    >
                        <div className="inline-flex items-center justify-center p-3 bg-blue-50 rounded-2xl mb-6 text-[#2563EB]">
                            <FileText className="w-8 h-8" />
                        </div>
                        <h1 className="text-4xl md:text-5xl font-extrabold text-[#0F172A] mb-6 font-jakarta">
                            WHOIS Domain Lookup
                        </h1>
                        <p className="text-lg text-[#475569] max-w-2xl mx-auto mb-10">
                            Find registration details, ownership, and expiry dates for any domain name.
                        </p>

                        {/* Search Bar */}
                        <div className="max-w-2xl mx-auto space-y-4">
                            <div className="bg-white p-2 rounded-2xl shadow-[0_20px_40px_-15px_rgba(0,0,0,0.05)] border border-[#E2E8F0] flex gap-2">
                                <input
                                    type="text"
                                    placeholder="Enter domain (e.g., google.com)"
                                    value={domain}
                                    onChange={(e) => setDomain(e.target.value)}
                                    onKeyDown={(e) => e.key === "Enter" && handleSearch()}
                                    className="flex-1 h-12 pl-4 pr-4 bg-transparent text-lg font-medium text-[#0F172A] placeholder-slate-400 outline-none"
                                    suppressHydrationWarning
                                />
                                <button
                                    onClick={(e) => handleSearch(e)}
                                    disabled={isLoading}
                                    className="h-12 px-8 bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white font-bold rounded-xl transition-all shadow-lg active:scale-95 disabled:opacity-70 disabled:cursor-not-allowed flex items-center gap-2"
                                >
                                    {isLoading ? <Loader2 className="w-5 h-5 animate-spin" /> : <Search className="w-5 h-5" />}
                                    <span className="hidden sm:inline">Lookup</span>
                                </button>
                            </div>

                            {/* Recent Searches */}
                            {recentSearches.length > 0 && (
                                <div className="flex flex-wrap items-center justify-center gap-2">
                                    <span className="text-xs font-semibold text-slate-400 uppercase tracking-wider flex items-center gap-1">
                                        <Clock className="w-3 h-3" /> Recent:
                                    </span>
                                    {recentSearches.map((search) => (
                                        <button
                                            key={search}
                                            onClick={() => handleSearch(undefined, search)}
                                            className="group flex items-center gap-1.5 bg-white border border-slate-200 px-3 py-1 rounded-full text-xs font-medium text-slate-600 hover:border-blue-300 hover:text-blue-600 transition-colors shadow-sm"
                                        >
                                            {search}
                                            <span
                                                onClick={(e) => removeFromHistory(e, search)}
                                                className="hover:bg-red-50 hover:text-red-500 rounded-full p-0.5 transition-colors"
                                            >
                                                <X className="w-3 h-3" />
                                            </span>
                                        </button>
                                    ))}
                                </div>
                            )}
                        </div>
                    </motion.div>
                </div>
            </section>

            {/* Content Area */}
            <main className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12 w-full relative z-10">
                {isLoading && (
                    <div className="w-full max-w-5xl mx-auto space-y-8 animate-pulse">
                        <div className="h-48 bg-slate-100 rounded-3xl w-full"></div>
                        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                            <div className="h-96 bg-slate-100 rounded-2xl"></div>
                            <div className="lg:col-span-2 h-96 bg-slate-100 rounded-2xl"></div>
                        </div>
                    </div>
                )}

                {error && !isLoading && (
                    <div className="max-w-xl mx-auto bg-red-50 border border-red-100 rounded-xl p-6 flex flex-col items-center text-center gap-3 text-red-700 animate-in fade-in zoom-in duration-300">
                        <div className="p-3 bg-red-100 rounded-full">
                            <AlertCircle className="w-8 h-8 text-red-600" />
                        </div>
                        <h3 className="text-lg font-bold text-red-900">Lookup Failed</h3>
                        <span className="font-medium text-red-700/80">{error}</span>
                    </div>
                )}

                {data && !isLoading && <WhoisResult data={data} />}
            </main>


        </div>
    );
}
