"use client";

import React, { useState } from "react";
import {
    Calendar, Shield, Server, FileText, ChevronDown, ChevronUp,
    Globe, Clock, User, Building, Phone, Mail, MapPin, Link as LinkIcon
} from "lucide-react";
import { motion, AnimatePresence } from "framer-motion";

export interface ContactInfo {
    name: string;
    organization: string;
    street: string;
    city: string;
    state: string;
    postalCode: string;
    country: string;
    phone: string;
    email: string;
}

export interface RegistrarInfo {
    name: string;
    ianaId: string;
    url: string;
    abuseContactEmail: string;
    abuseContactPhone: string;
}

export interface WhoisData {
    domainName: string;
    registrar: RegistrarInfo;
    whoisServer: string;
    updatedDate: string;
    creationDate: string;
    expirationDate: string;
    nameservers: string[];
    status: string[];
    dnssec: string;
    registrant: ContactInfo;
    administrative: ContactInfo;
    technical: ContactInfo;
    billing: ContactInfo;
    rawData: string;
}

interface WhoisResultProps {
    data: WhoisData;
}

export const WhoisResult: React.FC<WhoisResultProps> = ({ data }) => {
    const [isRawOpen, setIsRawOpen] = useState(false);
    const [activeTab, setActiveTab] = useState<'registrant' | 'admin' | 'tech' | 'billing'>('registrant');
    const [copiedField, setCopiedField] = useState<string | null>(null);

    const getFlagEmoji = (countryCode: string) => {
        if (!countryCode) return null;
        if (countryCode.length === 2) {
            return countryCode
                .toUpperCase()
                .replace(/./g, (char) => String.fromCodePoint(127397 + char.charCodeAt(0)));
        }
        return null;
    };

    const copyToClipboard = (text: string, fieldId: string) => {
        navigator.clipboard.writeText(text);
        setCopiedField(fieldId);
        setTimeout(() => setCopiedField(null), 1500);
    };

    const downloadReport = () => {
        const element = document.createElement("a");
        const file = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
        element.href = URL.createObjectURL(file);
        element.download = `${data.domainName}-whois-report.json`;
        document.body.appendChild(element);
        element.click();
    };

    const formatDate = (dateStr: string) => {
        if (!dateStr) return "N/A";
        try {
            return new Date(dateStr).toLocaleDateString("en-US", {
                year: 'numeric',
                month: 'long',
                day: 'numeric'
            });
        } catch {
            return dateStr;
        }
    };

    // Calculate Domain Age and Expiry
    const calculateTime = () => {
        if (!data.creationDate || !data.expirationDate) return null;
        const now = new Date();
        const created = new Date(data.creationDate);
        const expires = new Date(data.expirationDate);

        const ageDiff = Math.abs(now.getTime() - created.getTime());
        const ageDays = Math.ceil(ageDiff / (1000 * 60 * 60 * 24));
        const ageYears = Math.floor(ageDays / 365);

        const expiryDiff = expires.getTime() - now.getTime();
        const expiryDays = Math.ceil(expiryDiff / (1000 * 60 * 60 * 24));

        return { ageYears, ageDays: ageDays % 365, expiryDays };
    };

    const timeMetrics = calculateTime();

    const ContactCard = ({ info, title }: { info: ContactInfo, title: string }) => {
        if (!info || (!info.name && !info.organization)) {
            return (
                <div className="text-center py-12 text-slate-400 italic bg-slate-50/50 rounded-xl border border-dashed border-slate-200">
                    <User className="w-8 h-8 mx-auto mb-3 text-slate-300" />
                    No {title} information available.
                </div>
            );
        }

        const Field = ({ icon: Icon, label, value, id }: any) => {
            if (!value) return null;
            return (
                <div className="group relative bg-slate-50 hover:bg-white p-3 rounded-lg border border-transparent hover:border-slate-200 transition-all">
                    <div className="flex items-center gap-2 mb-1">
                        <Icon className="w-3.5 h-3.5 text-blue-500" />
                        <span className="text-xs font-semibold text-slate-400 uppercase tracking-wide">{label}</span>
                    </div>
                    <div className="flex items-center justify-between gap-4">
                        <span className="text-sm font-medium text-slate-800 truncate select-all">{value}</span>
                        <button
                            onClick={() => copyToClipboard(value, id)}
                            className="bg-white p-1.5 rounded-md shadow-sm border border-slate-100 text-slate-400 opacity-0 group-hover:opacity-100 transition-opacity hover:text-blue-600 active:scale-95"
                            title="Copy"
                        >
                            {copiedField === id ? <span className="text-[10px] font-bold text-green-600">Copied</span> : <LinkIcon className="w-3 h-3" />}
                        </button>
                    </div>
                </div>
            );
        };

        return (
            <div className="space-y-4 animate-in fade-in zoom-in duration-300">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <Field icon={User} label="Name" value={info.name} id={`name-${title}`} />
                    <Field icon={Building} label="Organization" value={info.organization} id={`org-${title}`} />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <Field icon={Mail} label="Email" value={info.email} id={`email-${title}`} />
                    <Field icon={Phone} label="Phone" value={info.phone} id={`phone-${title}`} />
                </div>

                <div className="bg-slate-50 rounded-lg p-4 border border-transparent hover:border-slate-200 transition-colors">
                    <div className="flex items-center gap-2 mb-2">
                        <MapPin className="w-3.5 h-3.5 text-blue-500" />
                        <span className="text-xs font-semibold text-slate-400 uppercase tracking-wide">Address</span>
                    </div>
                    <div className="text-sm text-slate-700 space-y-0.5">
                        {info.street && <div>{info.street}</div>}
                        <div>{[info.city, info.state, info.postalCode].filter(Boolean).join(", ")}</div>
                        {info.country && (
                            <div className="font-semibold flex items-center gap-2 mt-1">
                                {getFlagEmoji(info.country) && <span className="text-lg">{getFlagEmoji(info.country)}</span>}
                                {info.country}
                            </div>
                        )}
                    </div>
                </div>
            </div>
        );
    };

    return (
        <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            className="w-full max-w-6xl mx-auto space-y-6"
        >
            {/* Pro Header Dashboard */}
            <div className="bg-white border border-slate-200 rounded-3xl p-6 md:p-8 shadow-[0_2px_20px_-10px_rgba(0,0,0,0.1)]">
                <div className="flex flex-col xl:flex-row xl:items-start justify-between gap-8">
                    <div className="flex-1 min-w-0">
                        <div className="flex items-center gap-3 mb-2">
                            <span className="bg-blue-600 text-white text-[10px] font-bold px-2 py-0.5 rounded-full uppercase tracking-wider">Domain</span>
                            {data.dnssec === "signedDelegation" && (
                                <span className="bg-green-100 text-green-700 text-[10px] font-bold px-2 py-0.5 rounded-full uppercase tracking-wider flex items-center gap-1">
                                    <Shield className="w-3 h-3" /> Secure
                                </span>
                            )}
                        </div>
                        <h2 className="text-3xl md:text-5xl font-extrabold text-slate-900 font-jakarta break-all tracking-tight leading-tight">
                            {data.domainName}
                        </h2>

                        <div className="flex flex-wrap gap-2 mt-4">
                            {data.status && data.status.length > 0 ? (
                                data.status.slice(0, 4).map((status, i) => (
                                    <span key={i} className="bg-slate-100 text-slate-600 text-xs px-3 py-1.5 rounded-lg border border-slate-200 font-mono font-medium truncate max-w-[200px]" title={status}>
                                        {status.split(' ')[0]}
                                    </span>
                                ))
                            ) : (
                                <span className="text-slate-400 text-sm italic">Status not available</span>
                            )}
                        </div>
                    </div>

                    {/* Metrics Cards */}
                    {timeMetrics && (
                        <div className="flex gap-4 min-w-fit">
                            <div className="bg-blue-50 border border-blue-100 rounded-2xl p-5 w-40 flex flex-col items-center justify-center text-center shadow-sm">
                                <span className="text-blue-600 text-xs font-bold uppercase tracking-wider mb-1">Domain Age</span>
                                <div className="text-3xl font-extrabold text-blue-900">
                                    {timeMetrics.ageYears}<span className="text-sm font-medium text-blue-500 ml-0.5">y</span>
                                </div>
                                <span className="text-blue-600/70 text-xs mt-1">{timeMetrics.ageDays} days</span>
                            </div>

                            <div className={`bg-white border rounded-2xl p-5 w-40 flex flex-col items-center justify-center text-center shadow-sm ${timeMetrics.expiryDays < 30 ? 'border-red-200 bg-red-50' : 'border-slate-200'}`}>
                                <span className={`text-xs font-bold uppercase tracking-wider mb-1 ${timeMetrics.expiryDays < 30 ? 'text-red-600' : 'text-slate-400'}`}>Expires In</span>
                                <div className={`text-3xl font-extrabold ${timeMetrics.expiryDays < 30 ? 'text-red-900' : 'text-slate-900'}`}>
                                    {timeMetrics.expiryDays}
                                </div>
                                <span className={`${timeMetrics.expiryDays < 30 ? 'text-red-600/70' : 'text-slate-400'} text-xs mt-1`}>days</span>
                            </div>
                        </div>
                    )}
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mt-10 pt-8 border-t border-slate-100">
                    <div className="flex items-start gap-4">
                        <div className="p-2 bg-blue-50 rounded-lg text-blue-600"><Calendar className="w-5 h-5" /></div>
                        <div>
                            <p className="text-xs font-bold text-slate-400 uppercase tracking-wide">Registered On</p>
                            <p className="text-sm font-bold text-slate-900 mt-1">{formatDate(data.creationDate)}</p>
                        </div>
                    </div>
                    <div className="flex items-start gap-4">
                        <div className="p-2 bg-purple-50 rounded-lg text-purple-600"><Clock className="w-5 h-5" /></div>
                        <div>
                            <p className="text-xs font-bold text-slate-400 uppercase tracking-wide">Expires On</p>
                            <p className="text-sm font-bold text-slate-900 mt-1">{formatDate(data.expirationDate)}</p>
                        </div>
                    </div>
                    <div className="flex items-start gap-4">
                        <div className="p-2 bg-slate-100 rounded-lg text-slate-600"><FileText className="w-5 h-5" /></div>
                        <div>
                            <p className="text-xs font-bold text-slate-400 uppercase tracking-wide">Last Updated</p>
                            <p className="text-sm font-bold text-slate-900 mt-1">{formatDate(data.updatedDate)}</p>
                        </div>
                    </div>
                </div>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-12 gap-6">

                {/* Information Sidebar */}
                <div className="lg:col-span-4 space-y-6">
                    {/* Registrar Card */}
                    <div className="bg-white border border-slate-200 rounded-2xl p-6 shadow-sm overflow-hidden relative">
                        <div className="absolute top-0 right-0 w-24 h-24 bg-gradient-to-bl from-blue-50 to-transparent rounded-bl-full -mr-4 -mt-4 opacity-50" />
                        <h3 className="text-slate-900 font-bold mb-6 flex items-center gap-2 relative z-10">
                            <Shield className="w-5 h-5 text-blue-600 fill-blue-600/10" />
                            Registrar
                        </h3>

                        <div className="space-y-5 text-sm relative z-10">
                            <div>
                                <p className="text-[10px] font-bold text-slate-400 uppercase tracking-widest mb-1.5">Provider</p>
                                <p className="font-bold text-slate-900 text-lg leading-snug">{data.registrar?.name || "N/A"}</p>
                                {data.registrar?.ianaId && (
                                    <span className="inline-block mt-1 bg-slate-100 text-slate-500 text-[10px] px-2 py-0.5 rounded font-mono">IANA ID: {data.registrar.ianaId}</span>
                                )}
                            </div>

                            <div className="h-px bg-slate-100" />

                            <div className="space-y-3">
                                {data.registrar?.url && (
                                    <a href={data.registrar.url} target="_blank" rel="noreferrer" className="flex items-center gap-3 text-slate-600 hover:text-blue-600 transition-colors group">
                                        <div className="w-8 h-8 rounded-full bg-slate-50 flex items-center justify-center group-hover:bg-blue-50 group-hover:text-blue-600 transition-colors">
                                            <Globe className="w-4 h-4" />
                                        </div>
                                        <span className="font-medium truncate flex-1">{data.registrar.url.replace(/^https?:\/\//, '')}</span>
                                        <LinkIcon className="w-3 h-3 opacity-0 group-hover:opacity-100 transition-opacity" />
                                    </a>
                                )}

                                {data.registrar?.abuseContactEmail && (
                                    <a href={`mailto:${data.registrar.abuseContactEmail}`} className="flex items-center gap-3 text-slate-600 hover:text-blue-600 transition-colors group">
                                        <div className="w-8 h-8 rounded-full bg-slate-50 flex items-center justify-center group-hover:bg-blue-50 group-hover:text-blue-600 transition-colors">
                                            <Mail className="w-4 h-4" />
                                        </div>
                                        <span className="font-medium truncate flex-1">{data.registrar.abuseContactEmail}</span>
                                    </a>
                                )}

                                {data.registrar?.abuseContactPhone && (
                                    <div className="flex items-center gap-3 text-slate-600">
                                        <div className="w-8 h-8 rounded-full bg-slate-50 flex items-center justify-center">
                                            <Phone className="w-4 h-4" />
                                        </div>
                                        <span className="font-medium truncate flex-1">{data.registrar.abuseContactPhone}</span>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>

                    {/* Nameservers Card */}
                    <div className="bg-white border border-slate-200 rounded-2xl p-6 shadow-sm">
                        <h3 className="text-slate-900 font-bold mb-4 flex items-center gap-2">
                            <Server className="w-5 h-5 text-indigo-600" />
                            Nameservers
                        </h3>
                        <div className="space-y-2">
                            {data.nameservers && data.nameservers.length > 0 ? (
                                data.nameservers.map((ns, idx) => (
                                    <div key={idx} className="bg-slate-50 hover:bg-slate-100 transition-colors px-3 py-2.5 rounded-lg text-xs font-mono text-slate-600 flex items-center gap-3 truncate group">
                                        <div className="w-1.5 h-1.5 rounded-full bg-indigo-400 group-hover:bg-indigo-600 transition-colors" />
                                        {ns}
                                    </div>
                                ))
                            ) : (
                                <p className="text-slate-400 text-sm italic">No name servers listed.</p>
                            )}
                        </div>
                    </div>

                    {/* Actions */}
                    <button
                        onClick={downloadReport}
                        className="w-full py-3 px-4 bg-slate-800 hover:bg-slate-900 text-white font-bold rounded-xl shadow-lg shadow-slate-200 transition-all flex items-center justify-center gap-2 active:scale-95"
                    >
                        <FileText className="w-4 h-4" /> Download JSON Report
                    </button>
                </div>

                {/* Contact Information Tabs */}
                <div className="lg:col-span-8 bg-white border border-slate-200 rounded-2xl p-6 md:p-8 shadow-sm flex flex-col h-fit min-h-[500px]">
                    <div className="flex items-center justify-between mb-8">
                        <h3 className="text-xl font-bold text-slate-900 flex items-center gap-2">
                            <User className="w-6 h-6 text-blue-600" />
                            Contact Details
                        </h3>
                    </div>

                    {/* Styled Tabs */}
                    <div className="flex p-1 bg-slate-100 rounded-xl mb-8 overflow-x-auto">
                        {(['registrant', 'admin', 'tech', 'billing'] as const).map((tab) => (
                            <button
                                key={tab}
                                onClick={() => setActiveTab(tab)}
                                className={`flex-1 min-w-[100px] py-2.5 px-4 text-sm font-bold rounded-lg transition-all capitalize ${activeTab === tab
                                    ? "bg-white text-blue-600 shadow-sm"
                                    : "text-slate-500 hover:text-slate-700 hover:bg-slate-200/50"
                                    }`}
                            >
                                {tab === 'admin' ? 'Admin' : tab === 'tech' ? 'Technical' : tab}
                            </button>
                        ))}
                    </div>

                    <div className="flex-1">
                        <AnimatePresence mode="wait">
                            <motion.div
                                key={activeTab}
                                initial={{ opacity: 0, x: 10 }}
                                animate={{ opacity: 1, x: 0 }}
                                exit={{ opacity: 0, x: -10 }}
                                transition={{ duration: 0.2 }}
                            >
                                {activeTab === 'registrant' && <ContactCard info={data.registrant} title="Registrant" />}
                                {activeTab === 'admin' && <ContactCard info={data.administrative} title="Administrative" />}
                                {activeTab === 'tech' && <ContactCard info={data.technical} title="Technical" />}
                                {activeTab === 'billing' && <ContactCard info={data.billing} title="Billing" />}
                            </motion.div>
                        </AnimatePresence>
                    </div>
                </div>
            </div>

            {/* Raw Data Accordion (Professional) */}
            <div className="bg-white border border-slate-200 rounded-2xl shadow-sm overflow-hidden">
                <button
                    onClick={() => setIsRawOpen(!isRawOpen)}
                    className="w-full px-6 py-4 flex items-center justify-between bg-slate-50 hover:bg-slate-100 transition-colors group"
                >
                    <span className="font-bold text-slate-600 group-hover:text-slate-900 flex items-center gap-2 transition-colors">
                        <FileText className="w-4 h-4" /> View Raw WHOIS Response
                    </span>
                    {isRawOpen ? <ChevronUp className="w-5 h-5 text-slate-400" /> : <ChevronDown className="w-5 h-5 text-slate-400" />}
                </button>

                <AnimatePresence>
                    {isRawOpen && (
                        <motion.div
                            initial={{ height: 0 }}
                            animate={{ height: "auto" }}
                            exit={{ height: 0 }}
                            transition={{ duration: 0.2 }}
                            className="bg-slate-900"
                        >
                            <div className="p-6 relative">
                                <button
                                    onClick={(e) => { e.stopPropagation(); copyToClipboard(data.rawData, 'raw'); }}
                                    className="absolute top-4 right-4 bg-slate-800 text-slate-400 px-3 py-1.5 rounded-lg text-xs font-bold hover:bg-slate-700 hover:text-white transition-colors"
                                >
                                    {copiedField === 'raw' ? 'Copied!' : 'Copy Raw'}
                                </button>
                                <pre className="text-slate-300 font-mono text-xs overflow-x-auto whitespace-pre-wrap leading-relaxed max-h-[500px] scrollbar-thin scrollbar-thumb-slate-700 scrollbar-track-transparent">
                                    {data.rawData}
                                </pre>
                            </div>
                        </motion.div>
                    )}
                </AnimatePresence>
            </div>
        </motion.div>
    );
};
